<?php
/**
 * Eltron Admin page basic functions
 *
 * @package Eltron
 */

// Prevent direct access.
if ( ! defined( 'ABSPATH' ) ) exit;

class Eltron_Admin {
	/**
	 * Singleton instance
	 *
	 * @var Eltron_Admin
	 */
	private static $instance;

	/**
	 * Parent menu slug of all theme pages
	 *
	 * @var string
	 */
	private $_menu_id = 'eltron';

	/**
	 * ====================================================
	 * Singleton & constructor functions
	 * ====================================================
	 */

	/**
	 * Get singleton instance.
	 *
	 * @return Eltron_Admin
	 */
	public static function instance() {
		if ( is_null( self::$instance ) ) {
			self::$instance = new self();
		}
		return self::$instance;
	}

	/**
	 * Class constructor
	 */
	protected function __construct() {
		// General admin hooks on every admin pages
		add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_admin_styles' ) );
		add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_admin_javascripts' ) );

		add_action( 'admin_notices', array( $this, 'add_theme_welcome' ), 999 );

		// Classic editor hooks
		add_action( 'admin_init', array( $this, 'add_editor_css' ) );
		add_filter( 'tiny_mce_before_init', array( $this, 'add_classic_editor_custom_css' ) );
		add_filter( 'tiny_mce_before_init', array( $this, 'add_classic_editor_body_class' ) );
		add_filter( 'block_editor_settings', array( $this, 'add_gutenberg_custom_css' ) );

		$this->_includes();
	}

	/**
	 * Include additional files.
	 */
	private function _includes() {
		require_once( ELTRON_INCLUDES_DIR . '/admin/class-eltron-admin-fields.php' );

		// Only include metabox on post add/edit page and term add/edit page.
		global $pagenow;
		if ( in_array( $pagenow, array( 'post.php', 'post-new.php', 'edit-tags.php', 'term.php' ) ) ) {
			//require_once( ELTRON_INCLUDES_DIR . '/admin/class-eltron-admin-metabox-page-settings.php' );
		}
	}

	/**
	 * ====================================================
	 * Hook functions
	 * ====================================================
	 */

	/**
	 * Enqueue admin styles.
	 *
	 * @param string $hook
	 */
	public function enqueue_admin_styles( $hook ) {
		/**
		 * Hook: Styles to be included before admin CSS
		 */
		do_action( 'eltron/admin/before_enqueue_admin_css', $hook );

		// Register CSS files
		wp_register_style( 'alpha-color-picker', ELTRON_CSS_URL . '/vendors/alpha-color-picker' . ELTRON_ASSETS_SUFFIX . '.css', array( 'wp-color-picker' ), ELTRON_VERSION );

		// Enqueue CSS files
		wp_enqueue_style( 'eltron-admin', ELTRON_CSS_URL . '/admin/admin.css', array(), ELTRON_VERSION );
		wp_style_add_data( 'eltron-admin', 'rtl', 'replace' );

		/**
		 * Hook: Styles to be included after admin CSS
		 */
		do_action( 'eltron/admin/after_enqueue_admin_css', $hook );
	}

	/**
	 * Enqueue admin javascripts.
	 *
	 * @param string $hook
	 */
	public function enqueue_admin_javascripts( $hook ) {
		// Fetched version from package.json
		$ver = array();

		/**
		 * Hook: Styles to be included before admin JS
		 */
		do_action( 'eltron/admin/before_enqueue_admin_js', $hook );

		// Register JS files
		wp_register_script( 'alpha-color-picker', ELTRON_JS_URL . '/vendors/alpha-color-picker' . ELTRON_ASSETS_SUFFIX . '.js', array( 'jquery', 'wp-color-picker' ), ELTRON_VERSION, true );

		// Enqueue JS files.
		wp_enqueue_script( 'eltron-admin', ELTRON_JS_URL . '/admin/admin' . ELTRON_ASSETS_SUFFIX . '.js', array( 'jquery' ), ELTRON_VERSION, true );

		/**
		 * Hook: Styles to be included after admin JS
		 */
		do_action( 'eltron/admin/after_enqueue_admin_js', $hook );
	}

	/**
	 * Add welcome panel on the Appearance > Themes page.
	 */
	public function add_theme_welcome() {
		if ( 'themes' !== get_current_screen()->id ) {
			return;
		}
		?>
		<div class="eltron-admin-themes-welcome notice is-dismissible">
			<h2><?php esc_html_e( 'Welcome to Eltron!', 'eltron' ); ?></h2>
			<p><?php esc_html_e( 'Thank you for installing Eltron! Please visit the theme dashboard for more info about Eltron features.', 'eltron' ); ?></p>
			<p><?php esc_html_e( 'Next step is to go to Appearance -> Install Plugins to install and activate required plugins. Please note Install Plugins menu exists only if there are some plugins to be activated.', 'eltron' ); ?></p>
		</div>
		<?php
	}

	/**
	 * Add CSS for editor page.
	 */
	public function add_editor_css() {
		add_editor_style( ELTRON_CSS_URL . '/admin/editor' . ELTRON_ASSETS_SUFFIX . '.css' );

		add_editor_style( Eltron_Customizer::instance()->generate_active_google_fonts_embed_url() );
	}

	/**
	 * Add custom CSS to classic editor.
	 *
	 * @param array $settings
	 * @return array
	 */
	public function add_classic_editor_custom_css( $settings ) {
		// Skip Gutenberg editor page.
		$current_screen = get_current_screen();
		if ( method_exists( $current_screen, 'is_block_editor' ) && $current_screen->is_block_editor() ) {
			return $settings;
		}

		global $post;

		if ( empty( $post ) ) {
			return $settings;
		}

		$css_array = array(
			'global' => array(),
		);

		// TinyMCE HTML
		$css_array['global']['html']['background-color'] = '#fcfcfc';

		// Typography
		$active_google_fonts = array();
		$typography_types = array(
			'body' => 'body',
			'blockquote' => 'blockquote',
			'h1' => 'h1',
			'h2' => 'h2',
			'h3' => 'h3',
			'h4' => 'h4',
		);
		$fonts = eltron_get_all_fonts();

		foreach ( $typography_types as $type => $selector ) {
			// Font Family
			$font_family = eltron_get_theme_mod( $type . '_font_family' );
			$font_stack = $font_family;
			if ( '' !== $font_family && 'inherit' !== $font_family ) {
				$chunks = explode( '|', $font_family );
				if ( 2 === count( $chunks ) ) {
					$font_stack = eltron_array_value( $fonts[ $chunks[0] ], $chunks[1], $chunks[1] );
				}
			}
			if ( ! empty( $font_stack ) ) {
				$css_array['global'][ $selector ]['font-family'] = $font_stack;
			}

			// Font weight
			$font_weight = eltron_get_theme_mod( $type . '_font_weight' );
			if ( ! empty( $font_weight ) ) {
				$css_array['global'][ $selector ]['font-weight'] = $font_weight;
			}
			
			// Font style
			$font_style = eltron_get_theme_mod( $type . '_font_style' );
			if ( ! empty( $font_style ) ) {
				$css_array['global'][ $selector ]['font-style'] = $font_style;
			}
			
			// Text transform
			$text_transform = eltron_get_theme_mod( $type . '_text_transform' );
			if ( ! empty( $text_transform ) ) {
				$css_array['global'][ $selector ]['text-transform'] = $text_transform;
			}

			// Font size
			$font_size = eltron_get_theme_mod( $type . '_font_size' );
			if ( ! empty( $font_size ) ) {
				$css_array['global'][ $selector ]['font-size'] = $font_size;
			}

			// Line height
			$line_height = eltron_get_theme_mod( $type . '_line_height' );
			if ( ! empty( $line_height ) ) {
				$css_array['global'][ $selector ]['line-height'] = $line_height;
			}

			// Letter spacing
			$letter_spacing = eltron_get_theme_mod( $type . '_letter_spacing' );
			if ( ! empty( $letter_spacing ) ) {
				$css_array['global'][ $selector ]['letter-spacing'] = $letter_spacing;
			}
		}

		// Content wrapper width for content layout with sidebar
		// $css_array['global']['body.eltron-editor-left-sidebar']['width'] =
		// $css_array['global']['body.eltron-editor-right-sidebar']['width'] = 'calc(' . eltron_get_content_width_by_layout() . 'px + 2rem)';

		// // Content wrapper width for narrow content layout
		// $css_array['global']['body.eltron-editor-narrow']['width'] = 'calc(' . eltron_get_content_width_by_layout( 'narrow' ) . 'px + 2rem)';

		// // Content wrapper width for full content layout
		// $css_array['global']['body.eltron-editor-wide']['width'] = 'calc(' . eltron_get_content_width_by_layout( 'wide' ) . 'px + 2rem)';

		// Build CSS string.
		// $styles = str_replace( '"', '\"', eltron_convert_css_array_to_string( $css_array ) );
		$styles = wp_slash( eltron_convert_css_array_to_string( $css_array ) );

		// Merge with existing styles or add new styles.
		if ( ! isset( $settings['content_style'] ) ) {
			$settings['content_style'] = $styles . ' ';
		} else {
			$settings['content_style'] .= ' ' . $styles . ' ';
		}

		return $settings;
	}

	/**
	 * Add body class to classic editor.
	 *
	 * @param array $settings
	 * @return array
	 */
	public function add_classic_editor_body_class( $settings ) {
		// Skip Gutenberg editor page.
		$current_screen = get_current_screen();
		if ( method_exists( $current_screen, 'is_block_editor' ) && $current_screen->is_block_editor() ) {
			return $settings;
		}
		
		global $post;

		if ( empty( $post ) ) {
			return $settings;
		}

		$class = 'eltron-editor-' . eltron_get_page_setting_by_post_id( 'content_layout', $post->ID );

		// Merge with existing classes or add new class.
		if ( ! isset( $settings['body_class'] ) ) {
			$settings['body_class'] = $class . ' ';
		} else {
			$settings['body_class'] .= ' ' . $class . ' ';
		}

		return $settings;
	}

	/**
	 * Add custom CSS to Gutenberg editor.
	 *
	 * @param array $settings
	 * @return array
	 */
	public function add_gutenberg_custom_css( $settings ) {
		$css_array = array();

		// Content width
		$css_array['global']['.wp-block']['max-width'] = 'calc(' . eltron_get_theme_mod( 'content_narrow_width' ) . ' + ' . '30px)';
		$css_array['global']['.wp-block[data-align="wide"]']['max-width'] = 'calc(' . eltron_get_theme_mod( 'container_width' ) . ' + ' . '30px)';
		$css_array['global']['.wp-block[data-align="full"]']['max-width'] = 'none';
		$css_array['global']['.editor-post-title__block .editor-post-title__input']['font-family'] = 'inherit';
		$css_array['global']['.editor-post-title__block .editor-post-title__input']['font-weight'] = 'inherit';
		$css_array['global']['.editor-post-title__block .editor-post-title__input']['font-style'] = 'inherit';
		$css_array['global']['.editor-post-title__block .editor-post-title__input']['text-transform'] = 'inherit';
		$css_array['global']['.editor-post-title__block .editor-post-title__input']['font-size'] = 'inherit';
		$css_array['global']['.editor-post-title__block .editor-post-title__input']['line-height'] = 'inherit';
		$css_array['global']['.editor-post-title__block .editor-post-title__input']['letter-spacing'] = 'inherit';

		// Typography
		$active_google_fonts = array();
		$typography_types = array(
			'body' => 'body',
			'blockquote' => 'blockquote',
			'h1' => 'h1, .editor-post-title__block .editor-post-title__input',
			'h2' => 'h2',
			'h3' => 'h3',
			'h4' => 'h4',
			'title' => '.editor-post-title__block .editor-post-title__input',
		);
		$fonts = eltron_get_all_fonts();

		foreach ( $typography_types as $type => $selector ) {
			// Font Family
			$font_family = eltron_get_theme_mod( $type . '_font_family' );
			$font_stack = $font_family;
			if ( '' !== $font_family && 'inherit' !== $font_family ) {
				$chunks = explode( '|', $font_family );
				if ( 2 === count( $chunks ) ) {
					$font_stack = eltron_array_value( $fonts[ $chunks[0] ], $chunks[1], $chunks[1] );
				}
			}
			if ( ! empty( $font_stack ) ) {
				$css_array['global'][ $selector ]['font-family'] = $font_stack;
			}

			// Font weight
			$font_weight = eltron_get_theme_mod( $type . '_font_weight' );
			if ( ! empty( $font_weight ) ) {
				$css_array['global'][ $selector ]['font-weight'] = $font_weight;
			}
			
			// Font style
			$font_style = eltron_get_theme_mod( $type . '_font_style' );
			if ( ! empty( $font_style ) ) {
				$css_array['global'][ $selector ]['font-style'] = $font_style;
			}
			
			// Text transform
			$text_transform = eltron_get_theme_mod( $type . '_text_transform' );
			if ( ! empty( $text_transform ) ) {
				$css_array['global'][ $selector ]['text-transform'] = $text_transform;
			}

			// Font size
			$font_size = eltron_get_theme_mod( $type . '_font_size' );
			if ( ! empty( $font_size ) ) {
				$css_array['global'][ $selector ]['font-size'] = $font_size;
			}

			// Line height
			$line_height = eltron_get_theme_mod( $type . '_line_height' );
			if ( ! empty( $line_height ) ) {
				$css_array['global'][ $selector ]['line-height'] = $line_height;
			}

			// Letter spacing
			$letter_spacing = eltron_get_theme_mod( $type . '_letter_spacing' );
			if ( ! empty( $letter_spacing ) ) {
				$css_array['global'][ $selector ]['letter-spacing'] = $letter_spacing;
			}
		}

		// Relative heading margin top
		$css_array['global']['h1, h2, h3, h4, h5, h6']['margin-top'] = 'calc( 2 * ' . eltron_get_theme_mod( 'body_font_size' ) . ') !important';

		// Add to settings array.
		$settings['styles']['eltron-custom'] = array(
			'css' => eltron_convert_css_array_to_string( $css_array ),
		);

		return $settings;
	}

}

Eltron_Admin::instance();