<?php
/**
 * Eltron Pro module: Header Mega Menu Admin page
 */

// Prevent direct access.
if ( ! defined( 'ABSPATH' ) ) exit;

class Eltron_Pro_Module_Header_Mega_Menu_Admin {

	/**
	 * Singleton instance
	 *
	 * @var Eltron_Pro_Module_Header_Mega_Menu_Admin
	 */
	private static $instance;

	/**
	 * ====================================================
	 * Singleton & constructor functions
	 * ====================================================
	 */

	/**
	 * Get singleton instance.
	 *
	 * @return Eltron_Pro_Module_Header_Mega_Menu_Admin
	 */
	public static function instance() {
		if ( is_null( self::$instance ) ) {
			self::$instance = new self();
		}
		return self::$instance;
	}

	/**
	 * Class constructor
	 */
	protected function __construct() {
		// Use custom walker to enable 'wp_nav_menu_item_custom_fields' hook.
		add_filter( 'wp_edit_nav_menu_walker', array( $this, 'set_edit_nav_menu_walker' ), 99 );

		// Add custom fields to menu item editor.
		add_action( 'wp_nav_menu_item_custom_fields', array( $this, 'render_fields' ), 10, 4 );

		// Add custom iframe page for mega menu settings.
		add_action( 'admin_menu', array( $this, 'register_admin_menu' ) );
		add_action( 'current_screen', array( $this, 'render_settings' ) );
	}
	
	/**
	 * ====================================================
	 * Hook functions
	 * ====================================================
	 */

	/**
	 * Replace default menu editor walker class.
	 *
	 * @param string $walker
	 * @return string
	 */
	public function set_edit_nav_menu_walker( $walker ) {
		$walker = 'Menu_Item_Custom_Fields_Walker';

		if ( ! class_exists( $walker ) ) {
			require_once( ELTRON_PRO_DIR . 'inc/lib/class-menu-item-custom-fields-walker.php' );
		}

		return $walker;
	}

	/**
	 * Add admin submenu page: Dashboard > Eltron Mega Menu
	 */
	public function register_admin_menu() {
		add_dashboard_page(
			'',
			'',
			'edit_theme_options',
			'eltron-mega-menu-settings'
		);
	}
	
	/**
	 * ====================================================
	 * Render functions
	 * ====================================================
	 */

	/**
	 * Print menu item settings.
	 */
	public function render_settings() {
		if ( empty( $_GET['page'] ) || 'eltron-mega-menu-settings' !== $_GET['page'] || empty( $_GET['menu_item_id'] ) ) { // WPCS: CSRF ok, input var ok.
			return;
		}

		if ( ! defined( 'IFRAME_REQUEST' ) ) {
			define( 'IFRAME_REQUEST', true );
		}

		if ( ! isset( $_GET['depth'] ) ) {
			$_GET['depth'] = '2';
		}

		$depth = intval( $_GET['depth'] );
		$menu_item_id = intval( $_GET['menu_item_id'] );

		/**
		 * Dequeue unnecessary scripts.
		 */
		wp_dequeue_style( 'admin-bar' );

		/**
		 * WordPress Administration Template Header.
		 */
		iframe_header( esc_html__( 'Mega Menu Settings', 'eltron-features' ) );
		?>
		<div class="wrap eltron-nav-menu-item-mega-menu-settings">
			<?php
			/**
			 * Process save data.
			 */

			if ( isset( $_POST['submit'] ) ) {
				check_admin_referer( 'eltron_mega_menu_settings', 'eltron_mega_menu_settings_nonce' );

				$excludes = array(
					'eltron_mega_menu_settings_nonce',
					'_wp_http_referer',
					'depth',
					'submit',
				);

				$data = array();

				foreach ( $_POST as $key => $value ) {
					if ( in_array( $key, $excludes ) ) {
						continue;
					}

					if ( '' === $value ) continue;

					// If value is 0 or 1, cast to integer.
					if ( '0' === $value || '1' === $value ) {
						$value = intval( $value );
					}
					
					$data[ $key ] = $value;
				}

				if ( 0 === intval( $_POST['depth'] ) ) {
					update_post_meta( $menu_item_id, '_menu_item_eltron_mega_menu_settings', $data );
				} else {
					update_post_meta( $menu_item_id, '_menu_item_eltron_mega_menu_sub_settings', $data );
				}

				?>
				<div class="notice inline notice-success">
					<p><?php esc_html_e( 'Settings saved!', 'eltron-features' ); ?></p>
				</div>
				<?php
			}
			?>
			<hr class="wp-header-end">

			<form action="" method="post">
				<?php
				$menu_item_parent_id = get_post_meta( $menu_item_id, '_menu_item_menu_item_parent', true );
				
				wp_nonce_field( 'eltron_mega_menu_settings', 'eltron_mega_menu_settings_nonce' );

				switch ( $depth ) {
					// Show form for 1st level menu item.
					case 0:
						$this->_render_settings__depth_0( $menu_item_id );
						break;

					// Show form for 2nd level menu item.
					case 1:
						$this->_render_settings__depth_1( $menu_item_id );
						break;
					
					// Show nothing for other level menu item.
					default:
						$this->_render_settings__notice( $menu_item_id );
						break;
				}
				?>

				<input type="hidden" name="depth" value="<?php echo esc_attr( $depth ); ?>">
			</form>
		</div>

		<?php
		/**
		 * WordPress Administration Template Footer.
		 */
		iframe_footer();
		exit;
	}

	/**
	 * Print top menu item settings form.
	 *
	 * @param integer $menu_item_id
	 */
	private function _render_settings__depth_0( $menu_item_id ) {
		$values = get_post_meta( $menu_item_id, '_menu_item_eltron_mega_menu_settings', true );
		?>
		<h2>
			<?php printf(
				/* translators: %s: theme name. */
				esc_html__( '%s Mega Menu - Container Settings', 'eltron-features' ),
				eltron_get_theme_info( 'name' )
			); ?>
		</h2>
		<div class="notice inline notice-info">
			<p><?php esc_html_e( 'Please note, Mega Menu only works in desktop header (top bar, main bar, bottom bar).', 'eltron-features' ); ?></p>
		</div>
		<div class="eltron-admin-form">
			<?php $key = 'enabled'; ?>
			<div id="<?php echo esc_attr( 'eltron_mega_menu_' . $key ); ?>" class="eltron-admin-form-row">
				<div class="eltron-admin-form-label"><label><?php esc_html_e( 'Enable Mega Menu', 'eltron-features' ); ?></label></div>
				<div class="eltron-admin-form-field">
					<?php Eltron_Admin_Fields::render_field( array(
						'name'    => $key,
						'type'    => 'checkbox',
						'value'   => eltron_array_value( $values, $key ),
						'label'   => esc_attr__( 'Enable Mega Menu mode on this menu item (all direct child menu items will be columns)', 'eltron-features' ),
					) ); ?>
					<div class="notice notice-alt notice-info inline">
						<p><?php esc_html_e( 'You need to add at least 1 child menu item under this menu item.', 'eltron-features' ); ?></p>
					</div>
				</div>
			</div>

			<?php $key = 'size'; ?>
			<div id="<?php echo esc_attr( 'eltron_mega_menu_' . $key ); ?>" class="eltron-admin-form-row" <?php echo intval( eltron_array_value( $values, 'enabled' ) ) ? '' : 'style="display: none;"'; ?>>
				<div class="eltron-admin-form-label"><label><?php esc_html_e( 'Dropdown size', 'eltron-features' ); ?></label></div>
				<div class="eltron-admin-form-field">
					<?php Eltron_Admin_Fields::render_field( array(
						'name'    => $key,
						'type'    => 'select',
						'value'   => eltron_array_value( $values, $key ),
						'choices' => array(
							''       => esc_attr__( 'Same with container width', 'eltron-features' ),
							'custom' => esc_attr__( 'Use custom width', 'eltron-features' ),
						),
					) ); ?>
				</div>
			</div>

			<?php $key = 'custom_width'; ?>
			<div id="<?php echo esc_attr( 'eltron_mega_menu_' . $key ); ?>" class="eltron-admin-form-row" <?php echo intval( eltron_array_value( $values, 'enabled' ) ) && 'custom' === eltron_array_value( $values, 'size' ) ? '' : 'style="display: none;"'; ?>>
				<div class="eltron-admin-form-label"><label><?php esc_html_e( 'Custom width', 'eltron-features' ); ?></label></div>
				<div class="eltron-admin-form-field">
					<?php Eltron_Admin_Fields::render_field( array(
						'name'    => $key,
						'type'    => 'number',
						'min'     => 300,
						'value'   => eltron_array_value( $values, $key ),
					) ); ?>px
				</div>
			</div>
		</div>

		<?php submit_button(); ?>

		<script type="text/javascript">
			(function( $ ) {
				$(function() {
					var $container = $( '.eltron-nav-menu-item-mega-menu-settings' );

					$container.on( 'change', '#eltron_mega_menu_enabled input', function( e ) {
						if ( this.checked ) {
							$( '#eltron_mega_menu_size' ).show();

							if ( 'custom' === $( '#eltron_mega_menu_size select' ).val() ) {
								$( '#eltron_mega_menu_custom_width' ).show();
							}
						} else {
							$( '#eltron_mega_menu_size' ).hide();
							$( '#eltron_mega_menu_custom_width' ).hide();
						}
					});

					$container.on( 'change', '#eltron_mega_menu_size select', function( e ) {
						if ( 'custom' === this.value ) {
							$( '#eltron_mega_menu_custom_width' ).show();
						} else {
							$( '#eltron_mega_menu_custom_width' ).hide();
						}
					});
				});
			})(jQuery);
		</script>
		<?php
	}

	/**
	 * Print sub menu item settings form.
	 *
	 * @param integer $menu_item_id
	 */
	private function _render_settings__depth_1( $menu_item_id ) {
		$values = get_post_meta( $menu_item_id, '_menu_item_eltron_mega_menu_sub_settings', true );
		?>
		<h2>
			<?php printf(
				/* translators: %s: theme name. */
				esc_html__( '%s Mega Menu - Column Settings', 'eltron-features' ),
				eltron_get_theme_info( 'name' )
			); ?>
		</h2>
		<div class="notice inline notice-info">
			<p><?php esc_html_e( 'Please make sure you have enabled Mega Menu mode on parent menu item. Otherwise, these settings won\'t take effects.', 'eltron-features' ); ?></p>
		</div>
		<div class="eltron-admin-form">
			<?php $key = 'column_width'; ?>
			<div id="<?php echo esc_attr( 'eltron_mega_menu_' . $key ); ?>" class="eltron-admin-form-row">
				<div class="eltron-admin-form-label"><label><?php esc_html_e( 'Column width', 'eltron-features' ); ?></label></div>
				<div class="eltron-admin-form-field">
					<?php Eltron_Admin_Fields::render_field( array(
						'name'    => $key,
						'type'    => 'number',
						'min'     => 0,
						'max'     => 100,
						'step'    => 0.1,
						'value'   => eltron_array_value( $values, $key ),
					) ); ?>%
				</div>
			</div>

			<?php $key = 'hide_label'; ?>
			<div id="<?php echo esc_attr( 'eltron_mega_menu_' . $key ); ?>" class="eltron-admin-form-row">
				<div class="eltron-admin-form-label"><label><?php esc_html_e( 'Hide label', 'eltron-features' ); ?></label></div>
				<div class="eltron-admin-form-field">
					<?php Eltron_Admin_Fields::render_field( array(
						'name'    => $key,
						'type'    => 'checkbox',
						'value'   => eltron_array_value( $values, $key ),
						'label'   => esc_attr__( 'Hide column heading (navigation label)', 'eltron-features' ),
					) ); ?>
				</div>
			</div>

			<?php $key = 'disable_label_link'; ?>
			<div id="<?php echo esc_attr( 'eltron_mega_menu_' . $key ); ?>" class="eltron-admin-form-row">
				<div class="eltron-admin-form-label"><label><?php esc_html_e( 'Disable label link', 'eltron-features' ); ?></label></div>
				<div class="eltron-admin-form-field">
					<?php Eltron_Admin_Fields::render_field( array(
						'name'    => $key,
						'type'    => 'checkbox',
						'value'   => eltron_array_value( $values, $key ),
						'label'   => esc_attr__( 'Disable link on column heading (navigation label)', 'eltron-features' ),
					) ); ?>
				</div>
			</div>

			<?php $key = 'disable_padding'; ?>
			<div id="<?php echo esc_attr( 'eltron_mega_menu_' . $key ); ?>" class="eltron-admin-form-row">
				<div class="eltron-admin-form-label"><label><?php esc_html_e( 'Disable padding', 'eltron-features' ); ?></label></div>
				<div class="eltron-admin-form-field">
					<?php Eltron_Admin_Fields::render_field( array(
						'name'    => $key,
						'type'    => 'checkbox',
						'value'   => eltron_array_value( $values, $key ),
						'label'   => esc_attr__( 'Disable this column padding (might be useful when you use Gutenberg or page builders to build the custom content)', 'eltron-features' ),
					) ); ?>
				</div>
			</div>

			<?php $key = 'custom_content'; ?>
			<div id="<?php echo esc_attr( 'eltron_mega_menu_' . $key ); ?>" class="eltron-admin-form-row">
				<div class="eltron-admin-form-label"><label><?php esc_html_e( 'Custom content', 'eltron-features' ); ?></label></div>
				<div class="eltron-admin-form-field">
					<?php
					$value = get_post_meta( $menu_item_id, '_menu_item_' . $key, true );
					Eltron_Admin_Fields::render_field( array(
						'name'    => $key,
						'type'    => 'textarea',
						'rows'    => 3,
						'value'   => eltron_array_value( $values, $key ),
						'class'   => 'widefat',
					) );
					?>
					<p class="description"><?php esc_html_e( 'You can insert plain text, HTML tags, or shortcodes.', 'eltron-features' ); ?></p>
					<div class="notice notice-alt notice-info inline">
						<p><?php esc_html_e( 'TIPS: You can use our Custom Blocks module along with Gutenberg or other page builders to build more advanced and dynamic content. Create a new Custom Block, build the content, and then paste the Custom Block shortcode here.', 'eltron-features' ); ?></p>
					</div>
				</div>
			</div>
		</div>

		<?php submit_button();
	}

	/**
	 * Print notice on 3rd level menu items or higher.
	 *
	 * @param integer $menu_item_id
	 */
	private function _render_settings__notice( $menu_item_id ) {
		?>
		<div class="notice inline notice-warning">
			<p><?php esc_html_e( 'Mega Menu Settings are only available for 1st and 2nd level menu items.', 'eltron-features' ); ?></p>
		</div>
		<?php
	}

	/**
	 * Print mega item fields.
	 *
	 * @param integer $id
	 * @param WP_Post $item
	 * @param integer $depth
	 * @param array $args
	 */
	public function render_fields( $id, $item, $depth, $args ) {
		add_thickbox();

		$button_query = array(
			'page'         => 'eltron-mega-menu-settings',
			'menu_item_id' => $item->ID,
			'depth'        => 0,
			'TB_iframe'    => 'true',
			'width'        => 783,
			'height'       => 400,
		);
		?>
		<p class="eltron-nav-menu-item-mega-menu description-wide">
			<?php $button_query['depth'] = 0; ?>
			<a href="<?php echo esc_url( add_query_arg( $button_query, admin_url() ) ); ?>" class="button button-secondary eltron-nav-menu-item-mega-menu-button eltron-nav-menu-item-mega-menu-button-depth-0 thickbox">
				<?php printf(
					/* translators: %s: theme name. */
					esc_html__( '%s Mega Menu - Container Settings', 'eltron-features' ),
					eltron_get_theme_info( 'name' )
				); ?>
			</a>

			<?php $button_query['depth'] = 1; ?>
			<a href="<?php echo esc_url( add_query_arg( $button_query, admin_url() ) ); ?>" class="button button-secondary eltron-nav-menu-item-mega-menu-button eltron-nav-menu-item-mega-menu-button-depth-1 thickbox">
				<?php printf(
					/* translators: %s: theme name. */
					esc_html__( '%s Mega Menu - Column Settings', 'eltron-features' ),
					eltron_get_theme_info( 'name' )
				); ?>
			</a>
		</p>
		<?php
	}
}

Eltron_Pro_Module_Header_Mega_Menu_Admin::instance();