<?php
/**
 * Eltron Pro module: Custom Fonts Admin page
 */

// Prevent direct access.
if ( ! defined( 'ABSPATH' ) ) exit;

class Eltron_Pro_Module_Custom_Fonts_Admin {

	/**
	 * Singleton instance
	 *
	 * @var Eltron_Pro_Module_Custom_Fonts_Admin
	 */
	private static $instance;

	/**
	 * ====================================================
	 * Singleton & constructor functions
	 * ====================================================
	 */

	/**
	 * Get singleton instance.
	 *
	 * @return Eltron_Pro_Module_Custom_Fonts_Admin
	 */
	public static function instance() {
		if ( is_null( self::$instance ) ) {
			self::$instance = new self();
		}
		return self::$instance;
	}

	/**
	 * Class constructor
	 */
	protected function __construct() {
		// Upload mimes
		add_filter( 'upload_mimes', array( $this, 'add_upload_mimes' ) );

		// Settings API
		add_action( 'admin_init', array( $this, 'add_settings' ) );

		// Admin page
		add_action( 'eltron/admin/menu', array( $this, 'add_admin_menu' ) );
	}
	
	/**
	 * ====================================================
	 * Hook functions
	 * ====================================================
	 */

	/**
	 * Add upload mimes for font files.
	 *
	 * @param array $mimes
	 * @return array
	 */
	public function add_upload_mimes( $mimes ) {
		$mimes['otf'] = 'application/x-font-otf';
		$mimes['woff2'] = 'application/x-font-woff2';
		$mimes['woff'] = 'application/x-font-woff';
		$mimes['ttf'] = 'application/x-font-ttf';
		$mimes['eot'] = 'application/vnd.ms-fontobject';
		$mimes['svg'] = 'image/svg+xml';

		return $mimes;
	}

	/**
	 * Add admin submenu page: Appearance > Custom Fonts.
	 */
	public function add_admin_menu() {
		add_theme_page(
			esc_html__( 'Custom Fonts', 'eltron-features' ),
			esc_html__( 'Custom Fonts', 'eltron-features' ),
			'edit_theme_options',
			'eltron-custom-fonts',
			array( $this, 'render_admin_page' )
		);
	}

	/**
	 * Add settings via Options API.
	 */
	public function add_settings() {
		add_settings_section(
			'eltron_settings__custom_fonts',
			null,
			null,
			'eltron-custom-fonts'
		);

		add_settings_field(
			'eltron_custom_fonts',
			null,
			array( $this, 'render_setting__upload_custom_fonts' ),
			'eltron-custom-fonts',
			'eltron_settings__custom_fonts',
			array(
				'class' => 'hide-th',
			)
		);

		register_setting(
			'eltron-custom-fonts',
			'eltron_custom_fonts',
			array(
				'sanitize_callback' => array( $this, 'sanitize_setting__upload_custom_fonts' ),
				'default'           => array(),
			)
		);
	}

	/**
	 * ====================================================
	 * Render functions
	 * ====================================================
	 */

	/**
	 * Render admin page.
	 */
	public function render_admin_page() {
		?>
		<div class="wrap eltron-custom-fonts-wrap">
			<h1><?php echo get_admin_page_title(); ?></h1>
			<?php settings_errors(); ?>
			<hr class="wp-header-end">

			<form action="options.php" method="post">
				<?php
				settings_fields( 'eltron-custom-fonts' );
				do_settings_sections( 'eltron-custom-fonts' );
				submit_button();
				?>
			</form>
		</div>
		<?php
	}

	/**
	 * Render option field for "eltron_custom_fonts" setting.
	 */
	public function render_setting__upload_custom_fonts() {
		$fonts = get_option( 'eltron_custom_fonts', array() );

		$config = array(
			'defaultValues' => array(
				'name'     => '',
				'variants' => array( '400|normal' ),
			),
			'initEmpty' => 1 > count( $fonts ) ? true : false,
		);

		?>
		<p class="description"><?php esc_html_e( 'You can add as many custom fonts as you want. The uploaded fonts would be available on Customizer and Elementor Page Builder options for styling your typography.', 'eltron-features' ); ?></p>
		<p class="description"><?php esc_html_e( 'All the added fonts would be automatically enqueued on frontend, so please only add the fonts that you will use.', 'eltron-features' ); ?></p>
		<p class="description"><?php esc_html_e( 'Each font should have name and at least one variant, otherwise the font would not be saved.', 'eltron-features' ); ?></p>
		<p class="description"><?php esc_html_e( 'Please upload the font files in proper web font formats. You can use online service like Transfonter to convert your font into web font formats.', 'eltron-features' ); ?></p>
		<p class="description"><?php esc_html_e( 'The most essential format is .woff (and .woff2), because it has the best performance and supported by major browsers.', 'eltron-features' ); ?></p>
		<br>
		<div class="eltron-admin-repeater-control eltron-admin-custom-fonts eltron-admin-form metabox-holder" data-config="<?php echo esc_attr( json_encode( $config ) ); ?>">
			<ul data-repeater-list="eltron_custom_fonts" class="eltron-admin-repeater-control-list">
				<?php if ( 0 < count( $fonts ) ) {
					foreach ( $fonts as $i => $font ) {
						$this->render_setting__upload_custom_fonts__item( $font );
					}
				} else {
					$this->render_setting__upload_custom_fonts__item();
				} ?>
			</ul>
			<input data-repeater-create type="button" class="eltron-admin-repeater-control-add button" value="<?php esc_attr_e( 'Add Custom Font', 'eltron-features' ); ?>">
		</div>
		<?php
	}

	/**
	 * Render each item of "eltron_custom_fonts" setting field.
	 */
	public function render_setting__upload_custom_fonts__item( $font = array() ) {
		$is_template = false;

		if ( empty( $font ) ) {
			$is_template = true;
		}

		$font = wp_parse_args( $font, array(
			'name'     => '',
			'variants' => array(),
		) );

		$weights = array(
			'400' => esc_html__( 'Regular', 'eltron-features' ),
			'100' => esc_html__( 'Thin', 'eltron-features' ),
			'200' => esc_html__( 'Extra Light', 'eltron-features' ),
			'300' => esc_html__( 'Light', 'eltron-features' ),
			'500' => esc_html__( 'Medium', 'eltron-features' ),
			'600' => esc_html__( 'Semi Bold', 'eltron-features' ),
			'700' => esc_html__( 'Bold', 'eltron-features' ),
			'800' => esc_html__( 'Extra Bold', 'eltron-features' ),
			'900' => esc_html__( 'Black', 'eltron-features' ),
		);
		?>
		<li data-repeater-item class="eltron-admin-custom-font eltron-admin-repeater-control-item postbox" <?php echo $is_template ? 'style="display: none;"' : ''; ?>>
			
			<div class="eltron-admin-custom-font-name">
				<?php Eltron_Admin_Fields::render_field( array(
					'name'        => 'name',
					'type'        => 'text',
					'value'       => eltron_array_value( $font, 'name' ),
					'placeholder' => esc_attr__( 'Font name (e.g. Proxima Nova)', 'eltron-features' ),
					'class'       => 'widefat',
				) ); ?>
				<p class="description"><?php esc_html_e( 'The name of the font as it appears in the typography options.', 'eltron-features' ); ?></p>
			</div>

			<br>
			<h2><?php esc_html_e( 'Add variants and upload font files', 'eltron-features' ); ?></h2>

			<div class="eltron-admin-custom-font-variants">
				<?php
				foreach ( $weights as $weight => $weight_label ) :
					foreach ( array( 'normal', 'italic' ) as $style ) :
						$variant = $weight . '|' . $style;
						?>
						<div class="eltron-admin-custom-font-variant">
							<input type="checkbox" name="variants" value="<?php echo esc_attr( $variant ); ?>" <?php echo ( in_array( $variant, $font['variants'] ) ? 'checked' : '' ); ?>>
							<div class="postbox">
								<h2 class="hndle"><?php echo ( $weight_label . ' (' . $weight . ')' . ( 'italic' === $style ?  ' ' . esc_html__( 'Italic', 'eltron-features' ) : '' ) ); ?></h2>
								<div class="inside">
									<ul class="eltron-admin-font-variant-files">
										<?php foreach ( array( 'woff2', 'woff', 'ttf', 'eot', 'svg' ) as $extension ) : ?>
											<li class="eltron-admin-font-variant-file">
												<label><?php printf( '.%s', $extension ); ?></label>
												<?php Eltron_Admin_Fields::render_field( array(
													'name'        => 'file_' . $variant . '_' . $extension,
													'type'        => 'upload',
													'value'       => eltron_array_value( $font, 'file_' . $variant . '_' . $extension ),
													'class'       => 'widefat',
													'library'     => array( 'application', 'image' ),
													'frame_title' => sprintf( esc_html__( 'Select .%s file', 'eltron-features' ), $extension ),
												) ); ?>
											</li>
										<?php endforeach; ?>
									</ul>
								</div>
							</div>
						</div>
						<?php
					endforeach;
				endforeach;
				?>
			</div>

			<a href="javascript:;" data-repeater-delete type="button" class="eltron-admin-repeater-control-delete button button-small"><span class="dashicons dashicons-no-alt"></span></a>

		</li>
		<?php
	}

	/**
	 * ====================================================
	 * Sanitization functions
	 * ====================================================
	 */

	/**
	 * Sanitize custom fonts array input.
	 *
	 * @param array $data
	 * @return array
	 */
	public function sanitize_setting__upload_custom_fonts( $data ) {
		$sanitized = array();

		foreach ( $data as $i => $font ) {
			// Skip if font has no name.
			if ( empty( $font['name'] ) ) continue;
			
			// Skip if font has no variants.
			if ( empty( $font['variants'] ) ) continue;

			// Save non-empty value of current font data.
			$array = array();
			foreach ( $font as $key => $value ) {
				if ( '' !== $value ) {
					$array[ $key ] = $value;
				}
			}

			$sanitized[ $font['name'] ] = $array;
		}

		return $sanitized;
	}
}

Eltron_Pro_Module_Custom_Fonts_Admin::instance();