<?php
/**
 * Eltron Pro module: Sticky Header
 */

// Prevent direct access.
if ( ! defined( 'ABSPATH' ) ) exit;

class Eltron_Pro_Module_Header_Sticky extends Eltron_Pro_Module {

	/**
	 * Module name
	 *
	 * @var string
	 */
	const MODULE_SLUG = 'header-sticky';

	/**
	 * ====================================================
	 * Singleton & constructor functions
	 * ====================================================
	 */

	/**
	 * Class constructor
	 */
	protected function __construct() {
		parent::__construct();

		// Dynamic CSS
		add_filter( 'eltron/frontend/pro_dynamic_css', array( $this, 'add_dynamic_css' ) );

		// Customizer settings & values
		add_action( 'customize_register', array( $this, 'register_customizer_settings' ) );
		add_filter( 'eltron/customizer/setting_defaults', array( $this, 'add_customizer_setting_defaults' ) );
		add_filter( 'eltron/customizer/setting_postmessages', array( $this, 'add_customizer_setting_postmessages' ) );

		// Metabox
		add_action( 'eltron/admin/metabox/page_settings/fields', array( $this, 'add_metabox_page_settings_fields' ), 10, 2 );

		// Template filters
		$bar = eltron_get_theme_mod( 'header_sticky_bar' );
		add_filter( 'eltron/frontend/header_' . $bar . '_bar_attrs', array( $this, 'add_header_sticky_bar_attrs' ) );
		add_filter( 'eltron/frontend/header_' . $bar . '_bar_classes', array( $this, 'add_header_sticky_bar_classes' ) );
		add_action( 'eltron/frontend/logo', array( $this, 'render_header_sticky_logo' ), 20 );

		add_filter( 'eltron/frontend/header_mobile_main_bar_attrs', array( $this, 'add_header_mobile_sticky_bar_config' ) );
		add_filter( 'eltron/frontend/header_mobile_main_bar_classes', array( $this, 'add_header_mobile_sticky_bar_classes' ) );
		add_action( 'eltron/frontend/mobile_logo', array( $this, 'render_header_mobile_sticky_logo' ), 20 );

		// Page Settings
		add_filter( 'eltron/dataset/fallback_page_settings', array( $this, 'add_fallback_page_settings' ) );
	}
	
	/**
	 * ====================================================
	 * Hook functions
	 * ====================================================
	 */

	/**
	 * Add dynamic CSS from customizer settings into the inline CSS.
	 *
	 * @param string $css
	 * @return string
	 */
	public function add_dynamic_css( $css ) {
		// Skip adding dynamic CSS on customizer preview frame.
		if ( is_customize_preview() ) {
			return $css;
		}

		$postmessages = include( ELTRON_PRO_DIR . '/inc/modules/' . self::MODULE_SLUG . '/customizer/postmessages.php' );
		$defaults = include( ELTRON_PRO_DIR . '/inc/modules/' . self::MODULE_SLUG . '/customizer/defaults.php' );

		$css .= Eltron_Customizer::instance()->convert_postmessages_to_css_string( $postmessages, $defaults );

		return $css;
	}

	/**
	 * Register customizer sections, settings, and controls.
	 *
	 * @param WP_Customize_Manager $wp_customize
	 */
	public function register_customizer_settings( $wp_customize ) {
		$defaults = Eltron_Customizer::instance()->get_setting_defaults();

		require_once( ELTRON_PRO_DIR . 'inc/modules/' . self::MODULE_SLUG . '/customizer/options/_sections.php' );
		require_once( ELTRON_PRO_DIR . 'inc/modules/' . self::MODULE_SLUG . '/customizer/options/header--sticky.php' );
		require_once( ELTRON_PRO_DIR . 'inc/modules/' . self::MODULE_SLUG . '/customizer/options/page-settings.php' );
	}
	
	/**
	 * Add default values for all Customizer settings.
	 *
	 * @param array $defaults
	 * @return array
	 */
	public function add_customizer_setting_defaults( $defaults = array() ) {
		$add = include( ELTRON_PRO_DIR . 'inc/modules/' . self::MODULE_SLUG . '/customizer/defaults.php' );

		return array_merge_recursive( $defaults, $add );
	}

	/**
	 * Add postmessage rules for some Customizer settings.
	 *
	 * @param array $postmessages
	 * @return array
	 */
	public function add_customizer_setting_postmessages( $postmessages = array() ) {
		$add = include( ELTRON_PRO_DIR . 'inc/modules/' . self::MODULE_SLUG . '/customizer/postmessages.php' );

		return array_merge_recursive( $postmessages, $add );
	}

	/**
	 * Add Page Settings metabox fields.
	 */
	public function add_metabox_page_settings_fields( $obj, $tab ) {
		if ( 'header' !== $tab ) {
			return;
		}

		$option_key = 'eltron_page_settings';

		if ( is_a( $obj, 'WP_Post' ) ) {
			$values = get_post_meta( $obj->ID, '_' . $option_key, true );
		} elseif ( is_a( $obj, 'WP_Term' ) ) {
			$values = get_term_meta( $obj->term_id, $option_key, true );
		} else {
			$values = array();
		}

		?>
		<div class="eltron-admin-form-row">
			<div class="eltron-admin-form-label"><label><?php esc_html_e( 'Sticky desktop header', 'eltron-features' ); ?></label></div>
			<div class="eltron-admin-form-field">
				<?php
				$key = 'header_sticky';
				Eltron_Admin_Fields::render_field( array(
					'name'        => $option_key . '[' . $key . ']',
					'type'        => 'select',
					'choices'     => array(
						''  => esc_html__( '(Customizer)', 'eltron-features' ),
						'0' => esc_html__( 'No', 'eltron-features' ),
						'1' => esc_html__( 'Yes', 'eltron-features' ),
					),
					'value'       => eltron_array_value( $values, $key ),
				) );
				?>
			</div>
		</div>

		<div class="eltron-admin-form-row">
			<div class="eltron-admin-form-label"><label><?php esc_html_e( 'Sticky mobile header', 'eltron-features' ); ?></label></div>
			<div class="eltron-admin-form-field">
				<?php
				$key = 'header_mobile_sticky';
				Eltron_Admin_Fields::render_field( array(
					'name'        => $option_key . '[' . $key . ']',
					'type'        => 'select',
					'choices'     => array(
						''  => esc_html__( '(Customizer)', 'eltron-features' ),
						'0' => esc_html__( 'No', 'eltron-features' ),
						'1' => esc_html__( 'Yes', 'eltron-features' ),
					),
					'value'       => eltron_array_value( $values, $key ),
				) );
				?>
			</div>
		</div>
		<?php
	}

	/**
	 * Add sticky header configuration into tag's attributes.
	 *
	 * @param array $attrs
	 * @return array
	 */
	public function add_header_sticky_bar_attrs( $attrs ) {
		if ( intval( eltron_get_current_page_setting( 'header_sticky' ) ) ) {
			$bar = eltron_get_theme_mod( 'header_sticky_bar' );

			$attrs['data-sticky-height'] = esc_attr( min( intval( eltron_get_theme_mod( 'header_sticky_height' ) ), intval( eltron_get_theme_mod( 'header_' . $bar . '_bar_height' ) ) ) );
			$attrs['data-sticky-display'] = esc_attr( eltron_get_theme_mod( 'header_sticky_display' ) );
			$attrs['data-container-width'] = esc_attr( intval( eltron_get_theme_mod( 'container_width' ) ) );
		}

		return $attrs;
	}

	/**
	 * Add custom classes to the array of sticky header bar classes.
	 *
	 * @param array $classes
	 * @return array
	 */
	public function add_header_sticky_bar_classes( $classes ) {
		if ( intval( eltron_get_current_page_setting( 'header_sticky' ) ) ) {
			$classes['sticky'] = esc_attr( 'eltron-header-sticky' );
			$classes['sticky_display'] = esc_attr( 'eltron-header-sticky-display-' . eltron_get_theme_mod( 'header_sticky_display' ) );
		}

		return $classes;
	}

	/**
	 * Add sticky mobile header configuration into tag's attributes.
	 *
	 * @param array $attrs
	 * @return array
	 */
	public function add_header_mobile_sticky_bar_config( $attrs ) {
		if ( intval( eltron_get_current_page_setting( 'header_mobile_sticky' ) ) ) {
			$attrs['data-sticky-height'] = esc_attr( min( intval( eltron_get_theme_mod( 'header_mobile_sticky_height' ) ), intval( eltron_get_theme_mod( 'header_mobile_main_bar_height' ) ) ) );
			$attrs['data-sticky-display'] = esc_attr( eltron_get_theme_mod( 'header_mobile_sticky_display' ) );
		}

		return $attrs;
	}

	/**
	 * Add custom classes to the array of sticky mobile header bar classes.
	 *
	 * @param array $classes
	 * @return array
	 */
	public function add_header_mobile_sticky_bar_classes( $classes ) {
		if ( intval( eltron_get_current_page_setting( 'header_mobile_sticky' ) ) ) {
			$classes['sticky'] = esc_attr( 'eltron-header-sticky' );
			$classes['sticky_display'] = esc_attr( 'eltron-header-sticky-display-' . eltron_get_theme_mod( 'header_mobile_sticky_display' ) );
		}

		return $classes;
	}

	/**
	 * Modify fallback page settings.
	 *
	 * @param array $settings
	 * @return array
	 */
	public function add_fallback_page_settings( $settings ) {
		$settings['header_sticky'] = eltron_get_theme_mod( 'header_sticky', 0 );
		$settings['header_mobile_sticky'] = eltron_get_theme_mod( 'header_mobile_sticky', 0 );

		return $settings;
	}

	/**
	 * Render sticky logo.
	 */
	public function render_header_sticky_logo() {
		$logo_id = eltron_get_theme_mod( 'custom_logo_sticky', eltron_get_theme_mod( 'custom_logo' ) );
		$logo_width = intval( eltron_get_theme_mod( 'header_logo_width' ) );
		$sticky_logo_width = intval( eltron_get_theme_mod( 'header_sticky_logo_width', $logo_width ) );
		?>
		<span class="eltron-sticky-logo eltron-logo" data-width="<?php echo esc_attr( $logo_width ); ?>" data-sticky-width="<?php echo esc_attr( $sticky_logo_width ); ?>"><?php eltron_logo( $logo_id ); ?></span>
		<?php
	}

	/**
	 * Render sticky mobile logo.
	 */
	public function render_header_mobile_sticky_logo() {
		$logo_id = eltron_get_theme_mod( 'custom_logo_mobile_sticky', eltron_get_theme_mod( 'custom_logo_mobile' ) );
		$logo_width = intval( eltron_get_theme_mod( 'header_mobile_logo_width' ) );
		$sticky_logo_width = intval( eltron_get_theme_mod( 'header_mobile_sticky_logo_width', $logo_width ) );
		?>
		<span class="eltron-sticky-logo eltron-logo" data-width="<?php echo esc_attr( $logo_width ); ?>" data-sticky-width="<?php echo esc_attr( $sticky_logo_width ); ?>"><?php eltron_logo( $logo_id ); ?></span>
		<?php
	}
}

Eltron_Pro_Module_Header_Sticky::instance();