<?php
/**
 * Eltron Pro module: Custom Blocks (Hooks)
 */

// Prevent direct access.
if ( ! defined( 'ABSPATH' ) ) exit;

class Eltron_Pro_Module_Custom_Blocks extends Eltron_Pro_Module {

	/**
	 * Module name
	 *
	 * @var string
	 */
	const MODULE_SLUG = 'custom-blocks';

	/**
	 * Post type slug
	 *
	 * @var string
	 */
	const POST_TYPE = 'eltron_block';

	/**
	 * Array of active custom blocks on current loaded page.
	 *
	 * @var array
	 */
	private $rendered_custom_blocks = array();

	/**
	 * ====================================================
	 * Singleton & constructor functions
	 * ====================================================
	 */

	/**
	 * Class constructor
	 */
	protected function __construct() {
		parent::__construct();
		
		// Custom blocks post type
		add_action( 'init', array( $this, 'register_post_type' ) );

		// Frontend
		add_action( 'wp', array( $this, 'fetch_blocks_to_frontend' ), 999 );
		add_action( 'wp', array( $this, 'prevent_frontend_view_for_non_logged_in_users' ) );
		add_filter( 'single_template', array( $this, 'redirect_single_template' ) );

		// Shortcode
		add_shortcode( self::POST_TYPE, array( $this, 'shortcode' ) );

		// Admin page
		if ( is_admin() ) {
			require_once( ELTRON_PRO_DIR . 'inc/modules/' . self::MODULE_SLUG . '/class-eltron-pro-module-' . self::MODULE_SLUG . '-admin.php' );
		} elseif ( is_admin_bar_showing() ) {
			add_action( 'admin_bar_menu', array( $this, 'add_toolbar_menu' ), 80 );
			add_action( 'wp_enqueue_scripts', array( $this, 'add_toolbar_style' ) );
		}
	}
	
	/**
	 * ====================================================
	 * Hook functions
	 * ====================================================
	 */

	/**
	 * Register custom post type.
	 */
	public function register_post_type() {
		register_post_type( self::POST_TYPE, array(
			'labels'              => array(
				'name'                  => esc_html_x( 'Blocks', 'Post type general name', 'eltron-features' ),
				'singular_name'         => esc_html_x( 'Block', 'Post type singular name', 'eltron-features' ),
				'menu_name'             => esc_html_x( 'Blocks', 'Admin Menu text', 'eltron-features' ),
				'name_admin_bar'        => esc_html_x( 'Block', 'Add New on Toolbar', 'eltron-features' ),
				'add_new'               => esc_html__( 'Add New', 'eltron-features' ),
				'add_new_item'          => esc_html__( 'Add New Block', 'eltron-features' ),
				'new_item'              => esc_html__( 'New Block', 'eltron-features' ),
				'edit_item'             => esc_html__( 'Edit Block', 'eltron-features' ),
				'view_item'             => esc_html__( 'View Block', 'eltron-features' ),
				'all_items'             => esc_html__( 'All Blocks', 'eltron-features' ),
				'search_items'          => esc_html__( 'Search Blocks', 'eltron-features' ),
				'parent_item_colon'     => esc_html__( 'Parent Blocks:', 'eltron-features' ),
				'not_found'             => esc_html__( 'No Block found.', 'eltron-features' ),
				'not_found_in_trash'    => esc_html__( 'No Block found in Trash.', 'eltron-features' ),
				'featured_image'        => esc_html_x( 'Block Cover Image', 'Overrides the "Featured Image" phrase for this post type. Added in 4.3', 'eltron-features' ),
				'set_featured_image'    => esc_html_x( 'Set cover image', 'Overrides the "Set featured image" phrase for this post type. Added in 4.3', 'eltron-features' ),
				'remove_featured_image' => esc_html_x( 'Remove cover image', 'Overrides the "Remove featured image" phrase for this post type. Added in 4.3', 'eltron-features' ),
				'use_featured_image'    => esc_html_x( 'Use as cover image', 'Overrides the "Use as featured image" phrase for this post type. Added in 4.3', 'eltron-features' ),
				'archives'              => esc_html_x( 'Block archives', 'The post type archive label used in nav menus. Default "Post Archives". Added in 4.4', 'eltron-features' ),
				'insert_into_item'      => esc_html_x( 'Insert into Block', 'Overrides the "Insert into post"/"Insert into page" phrase (used when inserting media into a post). Added in 4.4', 'eltron-features' ),
				'uploaded_to_this_item' => esc_html_x( 'Uploaded to this Block', 'Overrides the "Uploaded to this post"/"Uploaded to this page" phrase (used when viewing media attached to a post). Added in 4.4', 'eltron-features' ),
				'filter_items_list'     => esc_html_x( 'Filter Block list', 'Screen reader text for the filter links heading on the post type listing screen. Default "Filter posts list"/"Filter pages list". Added in 4.4', 'eltron-features' ),
				'items_list_navigation' => esc_html_x( 'Blocks list navigation', 'Screen reader text for the pagination heading on the post type listing screen. Default "Posts list navigation"/"Pages list navigation". Added in 4.4', 'eltron-features' ),
				'items_list'            => esc_html_x( 'Blocks list', 'Screen reader text for the items list heading on the post type listing screen. Default "Posts list"/"Pages list". Added in 4.4', 'eltron-features' ),
			),
			'public'              => true,
			'exclude_from_search' => true,
			'publicly_queryable'  => true,
			'show_ui'             => true,
			'show_in_menu'        => false,
			'show_in_nav_menus'   => false,
			'rewrite'             => false,
			'supports'            => array( 'title', 'editor' ),
			'show_in_rest'        => true,
		) );
	}

	/**
	 * Fetch blocks attached to current page's hooks.
	 */
	public function fetch_blocks_to_frontend() {
		if ( is_admin() || is_singular( self::POST_TYPE ) ) return;

		// Iterate through each content block.
		// Check each display rules and attached hook.
		$all_blocks = get_posts( array(
			'post_type' => self::POST_TYPE,
			'posts_per_page' => -1,
		) );
		foreach ( $all_blocks as $block ) {
			if ( $this->check_block_status_on_current_page( $block ) ) {
				$settings = $this->get_block_settings( $block );

				// Get hook action.
				$hook_action = $settings['hook_action'];

				// Skip this block if no attached hook action specified.
				if ( empty( $hook_action ) ) {
					continue;
				}

				// More procedure for some specific actions.
				switch ( $hook_action ) {
					case 'custom':
						$hook_action = $settings['hook_action_custom'];
						break;

					case 'eltron/frontend/header':
						// Remove actions from original Eltron theme.
						remove_action( 'eltron/frontend/header', 'eltron_main_header' );
						remove_action( 'eltron/frontend/header', 'eltron_mobile_header' );
						remove_action( 'eltron/frontend/before_canvas', 'eltron_mobile_vertical_header' );

						// Remove actions from original Eltron Pro modules.
						remove_action( 'eltron/frontend/before_canvas', array( Eltron_Pro_Module_Header_Vertical::instance(), 'render_vertical_header' ) );
						break;

					case 'eltron/frontend/footer':
						// Remove actions from original Eltron Pro modules.
						remove_action( 'eltron/frontend/footer', 'eltron_main_footer' );
						break;
				}

				// Add this content block to the attached action.
				add_action( $hook_action, function() use( $block, $hook_action ) {
					$this->render_block( $block, $hook_action );
				}, $settings['hook_priority'] );

				// Add scripts of this content block.
				$this->enqueue_block_scripts( $block );
			}
		}
	}

	/**
	 * Don't display eltron_block post type on frontend.
	 */
	public function prevent_frontend_view_for_non_logged_in_users() {
		if ( is_singular( self::POST_TYPE ) && ! current_user_can( 'edit_posts' ) ) {
			wp_redirect( site_url(), 301 );
			die;
		}
	}

	/**
	 * Change single post template.
	 *
	 * @param  string $template
	 * @return string
	 */
	public function redirect_single_template( $template ) {
		global $post;

		if ( self::POST_TYPE === $post->post_type ) {
			$template = ELTRON_PRO_DIR . 'inc/modules/' . self::MODULE_SLUG . '/template.php';
		}

		return $template;
	}

	/**
	 * Add active custom blocks list to toolbar menu.
	 *
	 * @param WP_Admin_Bar $wp_admin_bar
	 */
	public function add_toolbar_menu( $wp_admin_bar ) {
		if ( ! current_user_can( 'edit_posts' ) ) {
			return;
		}

		$slug = 'eltron-custom-blocks';

		$wp_admin_bar->add_node( array(
			'id'     => $slug,
			'title'  => esc_html__( 'Eltron Custom Blocks', 'eltron-features' ),
		) );

		if ( 0 < count( $this->rendered_custom_blocks ) ) {
			foreach ( $this->rendered_custom_blocks as $item ) {
				$wp_admin_bar->add_node( array(
					'id'     => $slug . '--' . $item['id'],
					'parent' => $slug,
					'title'  => '<span class="eltron-toolbar-custom-block-title">' . $item['title'] . '</span><span class="eltron-toolbar-custom-block-meta">' . $item['hook'] . '</span>',
					'href'   => esc_url( add_query_arg( array( 'post' => $item['id'], 'action' => 'edit' ), admin_url( 'post.php' ) ) ),
				) );
			}
		} else {
			$wp_admin_bar->add_node( array(
				'id'     => $slug . '--empty',
				'parent' => $slug,
				'title'  => '<span class="eltron-toolbar-custom-block-meta">' . esc_html__( 'No Custom Block attached on this page.', 'eltron-features' ) . '</span>',
			) );
		}
	}

	/**
	 * Add style for active custom blocks list on toolbar menu.
	 */
	public function add_toolbar_style() {
		$css = '
		#wpadminbar #wp-admin-bar-eltron-custom-blocks > .ab-item:before {
			content: "\f180";
			top: 2px;
		}

		#wpadminbar #wp-admin-bar-eltron-custom-blocks .ab-submenu {
			white-space: nowrap;
		}

		#wpadminbar #wp-admin-bar-eltron-custom-blocks .ab-submenu .ab-item {
			height: auto;
			padding: 5px 10px;
		}

		#wpadminbar #wp-admin-bar-eltron-custom-blocks .eltron-toolbar-custom-block-title {
			display: block;
			font-size: inherit;
			line-height: 20px;
		}

		#wpadminbar #wp-admin-bar-eltron-custom-blocks .eltron-toolbar-custom-block-meta {
			display: block;
			font-size: 11px;
			font-style: italic;
			line-height: 14px;
			opacity: 0.5;
		}
		';
		wp_add_inline_style( 'admin-bar', eltron_minify_css_string( $css ) );
	}

	/**
	 * ====================================================
	 * Render functions
	 * ====================================================
	 */

	/**
	 * Render shortcode.
	 *
	 * @param array $args
	 * @return string
	 */
	public function shortcode( $args ) {
		$args = shortcode_atts( array(
			'id' => 0,
		), $args, self::POST_TYPE );

		if ( empty( $args['id'] ) ) {
			return;
		}

		if ( ! $this->check_block_status_on_current_page( $args['id'] ) ) {
			return;
		}

		// Render the custom block.
		ob_start();
		$this->render_block( $args['id'] );
		return ob_get_clean();
	}

	/**
	 * Enqueue custom block scripts.
	 *
	 * @param integer|WP_Post $block
	 */
	public function enqueue_block_scripts( $block ) {
		if ( ! is_a( $block, 'WP_Post' ) ) {
			$block = get_post( $block );
		}

		// Abort if there is no block with specified ID were found.
		if ( empty( $block ) ) {
			return;
		}

		// Abort if block is not yet published.
		if ( 'publish' !== get_post_status( $block ) ) {
			return;
		}

		// Get the editor/builder.
		$builder = $this->get_block_builder( $block );

		/**
		 * Get block scripts based on its editor/builder.
		 */

		switch ( $builder ) {
			case 'elementor':
				add_action( 'wp_enqueue_scripts', function() {
					// Enqueue Elementor scripts.
					\Elementor\Plugin::instance()->frontend->enqueue_styles();

					// Enqueue Elementor Pro scripts.
					if ( class_exists( '\ElementorPro\Plugin' ) ) {
						\ElementorPro\Plugin::instance()->enqueue_styles();
					}

					// Enqueue current page scripts.
					if ( class_exists( '\Elementor\Core\Files\CSS\Post' ) ) {
						$css_file = new \Elementor\Core\Files\CSS\Post( $block->ID );
					} elseif ( class_exists( '\Elementor\Post_CSS_File' ) ) {
						$css_file = new \Elementor\Post_CSS_File( $block->ID );
					}
					$css_file->enqueue();
				});
				break;

			case 'brizy':
				$brizy_post   = Brizy_Editor_Post::get( $block->ID );
				$brizy_public = new Brizy_Public_Main( $brizy_post );
				
				// Enqueue general Brizy scripts.
				add_action( 'wp_enqueue_scripts', array( $brizy_public, '_action_enqueue_preview_assets' ), 9999 );

				// Enqueue current page scripts.
				add_action( 'wp_head', function() use ( $brizy_post ) {
					$brizy_project = Brizy_Editor_Project::get();
					$brizy_html    = new Brizy_Editor_CompiledHtml( $brizy_post->get_compiled_html() );

					echo apply_filters( 'brizy_content', $brizy_html->get_head(), $brizy_project, $brizy_post->get_wp_post() );
				} );
				break;
		}
	}

	/**
	 * Render custom block.
	 *
	 * @param integer|WP_Post $block
	 * @param string $hook_action
	 */
	public function render_block( $block, $hook_action = null ) {
		if ( ! is_a( $block, 'WP_Post' ) ) {
			$block = get_post( $block );
		}

		// Abort if there is no block with specified ID were found.
		if ( empty( $block ) ) {
			return;
		}

		// Abort if block is not yet published.
		if ( 'publish' !== get_post_status( $block ) ) {
			return;
		}

		// Add this custom block to current loaded list.
		$this->add_rendered_custom_block( $block, $hook_action );

		// Get the editor/builder.
		$builder = $this->get_block_builder( $block );

		/**
		 * Check if the block needs a wrapper.
		 */

		// Use the `wp_body_open` hook to check current position. If it's hooked inside <body> tag, then it's safe to include the wrapper.
		if ( did_action( 'wp_body_open' ) ) {
			// Define array for wrapper classes.
			$wrapper_classes = array( 'eltron-block', 'eltron-block-' . $block->ID );

			// Add block responsive visibility class.
			$settings = $this->get_block_settings( $block );
			foreach ( array( 'desktop', 'tablet', 'mobile' ) as $device ) {
				if ( ! in_array( $device, $settings['devices'] ) ) {
					$wrapper_classes[] = 'eltron-hide-on-' . $device;
				}
			}

			// Add builder class.
			switch ( $builder ) {
				case 'elementor':
					$wrapper_classes[] = 'eltron-block-built-with-elementor';
					break;

				case 'brizy':
					$wrapper_classes[] = 'eltron-block-built-with-brizy';
					$wrapper_classes[] = 'brz'; // Mandatory Brizy content wrapper class, because all Brizy CSS use this selector wrapper.
					break;

				case 'divi-builder':
					$wrapper_classes[] = 'eltron-block-built-with-divi-builder';
					break;
			}

			// Add "added via shortcode" class.
			if ( is_null( $hook_action ) ) {
				$wrapper_classes[] = 'eltron-block-via-shortcode';
			}

			// Add custom additional classes.
			if ( ! empty( $settings['class'] ) ) {
				$wrapper_classes[] = $settings['class'];
			}

			// Wrapper tag.
			$before_block = '<' . $settings['tag'] . ( ! empty( $settings['id'] ) ? ' id="' . esc_attr( $settings['id'] ) . '"' : '' ) . ' class="' . esc_attr( implode( ' ', $wrapper_classes ) ) . '">';
			$after_block = '</' . $settings['tag'] . '>';

			// Whether to add content wrapper or not.
			if ( $settings['use_container'] ) {
				$before_block = $before_block . '<div class="eltron-wrapper eltron-block-wrapper"' . ( ! empty( $settings['container_width'] ) ? ' style="width: ' . $settings['container_width'] . 'px;"' : '' ) . '>';
				$after_block = '</div>' . $after_block;
			}
		}
		// Hooked on <head> tag, don't include wrapper.
		else {
			$before_block = '';
			$after_block = '';
		}

		/**
		 * Get block content based on its editor/builder.
		 */

		switch ( $builder ) {
			case 'elementor':
				// Fetch content using Elementor functions.
				$content = \Elementor\Plugin::instance()->frontend->get_builder_content_for_display( $block->ID );
				break;

			case 'brizy':
				$brizy_post = Brizy_Editor_Post::get( $block->ID );
				$brizy_project = Brizy_Editor_Project::get();
				$brizy_html = new Brizy_Editor_CompiledHtml( $brizy_post->get_compiled_html() );
				$brizy_public = new Brizy_Public_Main( $brizy_post ); // Mandatory, because it contains additional "brizy_content" filter.

				// Fetch content using Brizy filters.
				$content = apply_filters( 'brizy_content', $brizy_html->get_body(), $brizy_project, $brizy_post->get_wp_post() );

				// Print scripts manually if rendered via shortcode.
				// Because when the shortcode is processed, it's already too late to print Brizy scripts on <head>.
				if ( is_null( $hook_action ) ) {
					// Print block scripts at the bottom of <body>.
					// It should be rendered after Brizy's main CSS.
					// Can't find any better location to print the scripts.
					add_action( 'wp_print_footer_scripts', function() use ( $brizy_post, $brizy_project, $brizy_html ) {
						echo apply_filters( 'brizy_content', $brizy_html->get_head(), $brizy_project, $brizy_post->get_wp_post() );
					} );

					// Enqueue Brizy scripts.
					$brizy_public->_action_enqueue_preview_assets();
				}
				break;

			case 'divi-builder':
				global $wp_filter;

				// Temporarily switch the global $post to current block.
				global $post;
				$post = $block;

				// Add wrapper manually
				//
				// Can't use this call:
				// $content = et_builder_add_builder_content_wrapper( $block->post_content );
				//
				// Because 'et_builder_add_builder_content_wrapper' function is restricted only for singular template.
				$outer_class   = apply_filters( 'et_builder_outer_content_class', array( 'et-boc' ) );
				$outer_classes = implode( ' ', $outer_class );
				// $outer_id      = apply_filters( 'et_builder_outer_content_id', 'et-boc' );
				$inner_class   = apply_filters( 'et_builder_inner_content_class', array( 'et_builder_inner_content' ) );
				$inner_classes = implode( ' ', $inner_class );

				$is_dbp                   = et_is_builder_plugin_active();
				// $dbp_compat_wrapper_open  = $is_dbp ? '<div id="et_builder_outer_content" class="et_builder_outer_content">' : '';
				$dbp_compat_wrapper_open  = $is_dbp ? '<div class="et_builder_outer_content">' : '';
				$dbp_compat_wrapper_close = $is_dbp ? '</div>' : '';

				$content = sprintf(
					'<div class="%1$s">
						%2$s
						<div class="%3$s">
							%4$s
						</div>
						%5$s
					</div>',
					esc_attr( $outer_classes ),
					et_core_intentionally_unescaped( $dbp_compat_wrapper_open, 'fixed_string' ),
					esc_attr( $inner_classes ),
					$block->post_content,
					et_core_intentionally_unescaped( $dbp_compat_wrapper_close, 'fixed_string' )
				);

				// Apply the_content filters.
				$content = apply_filters( 'the_content', $block->post_content );

				// Revert back to original global $post.
				wp_reset_postdata();
				break;
			
			default:
				$content = $block->post_content;

				// Define filter functions for default content rendering.
				$functions = array(
					// 9
					'do_blocks',
					// 10
					'wptexturize',
					'wpautop',
					'shortcode_unautop',
					'prepend_attachment',
					'wp_make_content_images_responsive',
					// 11
					'capital_P_dangit',
					'do_shortcode',
					// 20
					'convert_smilies',
				);

				// Detect whether content is built using Gutenberg or Classic Editor.
				if ( function_exists( 'has_blocks' ) && has_blocks( $content ) ) {
					$removed_function = 'wpautop';
				} else {
					$removed_function = 'do_blocks';
				}

				// Remove unnecessary filter based on its editor type.
				$key = array_search( $removed_function, $functions );
				if ( $key !== false) {
					unset( $functions[$key] );
				}

				// Call filter functions one by one according to the order.
				foreach ( $functions as $function ) {
					if ( function_exists( $function ) ) {
						$content = $function( $content );
					}
				}
				break;
		}

		/**
		 * Render the block.
		 */

		// Render the markup.
		echo "\n" . '<!-- [eltron-block-' . $block->ID . '] -->' . $before_block . $content . $after_block . '<!-- [/eltron-block-' . $block->ID . '] -->' . "\n"; // WPCS: XSS OK.
	}

	/**
	 * ====================================================
	 * Private functions
	 * ====================================================
	 */

	/**
	 * Return all display rules for specified post.
	 *
	 * @param WP_Post $obj
	 * @return array
	 */
	public function get_current_page_rules__post( $obj ) {
		$return = array( 'general|global' );
			
		if ( $obj->ID == get_option( 'page_on_front' ) ) {
			$return[] = 'general|front_page';
		}

		$post_type = get_post_type( $obj );

		$return[] = 'general|singular';
		$return[] = 'posts|' . $post_type;
		$return[] = 'posts|' . $post_type . ':' . $obj->ID;
		
		$the_taxonomies = get_taxonomies( array(
			'object_type' => array( $post_type ),
			'public'      => true,
			'rewrite'     => true,
		) );

		foreach ( $the_taxonomies as $taxonomy ) {
			$terms_objects = get_the_terms( $obj, $taxonomy );

			if ( is_array( $terms_objects ) ) {
				foreach ( $terms_objects as $term_obj ) {
					$return[] = 'posts_in_tax|' . $taxonomy . ':' . $term_obj->term_id;
				}
			}
		}

		return $return;
	}

	/**
	 * Return all display rules for specified term.
	 *
	 * @param WP_Term $obj
	 * @return array
	 */
	public function get_current_page_rules__term( $obj ) {
		$return = array( 'general|global' );

		$return[] = 'general|archive';
		$return[] = 'tax_archive|' . $obj->taxonomy;
		$return[] = 'tax_archive|' . $obj->taxonomy . ':' . $obj->term_id;

		return $return;
	}

	/**
	 * Return all display rules for current loaded page.
	 *
	 * @return array
	 */
	public function get_current_page_rules() {
		$return = array( 'general|global' );
			
		if ( is_front_page() ) {
			$return[] = 'general|front_page';
		}

		if ( is_home() ) {
			$return[] = 'general|archive';
			$return[] = 'post_type_archive|post';
		}

		if ( is_singular() ) {
			$obj = get_queried_object();

			$return[] = 'general|singular';
			$return[] = 'posts|' . get_post_type();
			$return[] = 'posts|' . get_post_type() . ':' . get_the_ID();

			$the_taxonomies = get_taxonomies( array(
				'object_type' => array( get_post_type() ),
				'public'      => true,
				'rewrite'     => true,
			) );

			foreach ( $the_taxonomies as $taxonomy ) {
				$return[] = 'posts_in_tax|' . $taxonomy;

				$terms_objects = get_the_terms( $obj, $taxonomy );

				if ( is_array( $terms_objects ) ) {
					foreach ( $terms_objects as $term_obj ) {
						$return[] = 'posts_in_tax|' . $taxonomy . ':' . $term_obj->term_id;
					}
				}
			}
		}

		elseif ( is_archive() ) {
			$return[] = 'general|archive';

			if ( is_post_type_archive() ) {
				$obj = get_queried_object();
				
				$return[] = 'post_type_archive|' . $obj->name;
			}

			elseif ( is_tax() || is_category() || is_tag() ) {
				$obj = get_queried_object();

				$return[] = 'tax_archive|' . $obj->taxonomy;
				$return[] = 'tax_archive|' . $obj->taxonomy . ':' . $obj->term_id;
			}

			elseif ( is_year() || is_month() || is_date() || is_time() ) {
				$return[] = 'general|date_archive';
			}

			elseif ( is_author() ) {
				$return[] = 'general|author_archive';
			}
		}

		elseif ( is_search() ) {
			$return[] = 'general|search';
		}

		elseif ( is_404() ) {
			$return[] = 'general|not_found';
		}

		return $return;
	}

	/**
	 * Return all user roles for current loaded page.
	 *
	 * @return array
	 */
	public function get_current_user_roles() {
		$return = array( 'public' );

		if ( is_user_logged_in() ) {
			$return[] = 'logged_in';

			$user = wp_get_current_user();
			$return = array_merge( $return, $user->roles );
		} else {
			$return[] = 'logged_out';
		}

		return $return;
	}

	/**
	 * Render custom block.
	 *
	 * @param integer|WP_Post $block
	 * @param string $hook_action
	 */
	public function add_rendered_custom_block( $block, $hook_action = null ) {
		$hook_title = esc_html__( 'via Shortcode' );

		if ( ! is_null( $hook_action ) ) {
			$hook_names = $this->get_all_template_hooks( true );

			if ( isset( $hook_names['hook_action'] ) ) {
				$hook_title = $hook_names['hook_action'];
			} else {
				$hook_title = sprintf(
					/* translators: %1$s: custom hook name. */
					esc_html__( 'Hook: %1$s', 'eltron-features' ),
					$hook_action
				);
			}
		}

		$this->rendered_custom_blocks[ $block->ID ] = array(
			'id'    => $block->ID,
			'title' => get_the_title( $block ),
			'hook'  => $hook_title,
		);
	}

	/**
	 * Return the builder slug that is used to build the specified block id.
	 *
	 * @param integer|WP_Post $block
	 * @return string
	 */
	public function get_block_builder( $block ) {
		// Check if it's built with Elementor.
		if ( class_exists( '\Elementor\Plugin' ) && intval( \Elementor\Plugin::instance()->db->is_built_with_elementor( $block->ID ) ) ) {
			return 'elementor';
		}

		// Check if it's built with Brizy.
		elseif ( class_exists( 'Brizy_Editor_Post' ) && Brizy_Editor_Post::get( $block->ID )->uses_editor() ) {
			return 'brizy';
		}

		// Check if it's built with Divi.
		elseif ( class_exists( 'ET_Builder_Plugin' ) && et_pb_is_pagebuilder_used( $block->ID ) ) {
			return 'divi-builder';
		}

		return 'default';
	}

	/**
	 * Return settings of the specified block id.
	 *
	 * @param integer|WP_Post $block
	 * @return array
	 */
	public function get_block_settings( $block ) {
		if ( is_a( $block, 'WP_Post' ) ) {
			$block = $block->ID;
		}

		$settings = get_post_meta( $block, '_' . self::POST_TYPE . '_settings', true );
		$settings = wp_parse_args( $settings, array(
			'hook_action'        => '',
			'hook_action_custom' => '',
			'hook_priority'      => 10,

			'display_rules'      => array(),
			'exclusion_rules'    => array(),
			'user_roles'         => array(),
			'devices'            => array(),

			'tag'                => 'div',
			'id'                 => '',
			'class'              => '',
			'use_container'      => 0,
			'container_width'    => '',
		) );

		return $settings;
	}

	/**
	 * Return boolean whether the specififed block is available on current page or not.
	 *
	 * @param integer|WP_Post $block
	 * @return boolean
	 */
	public function check_block_status_on_current_page( $block ) {
		if ( ! is_a( $block, 'WP_Post' ) ) {
			$block = get_post( $block );
		}

		$settings = $this->get_block_settings( $block );

		// Fetch current page diplay rules.
		$current_page_rules = $this->get_current_page_rules();
		$current_user_roles = $this->get_current_user_roles();

		// Skip if content block doesn't have any assigned display rule.
		if ( empty( $settings['display_rules'] ) ) return false;

		// Skip if content block doesn't have any assigned user role.
		if ( empty( $settings['user_roles'] ) ) return false;

		// Check each display rule if matches current page.
		// Skip if there is no display rule matched.
		if ( 0 === count( array_intersect( $settings['display_rules'], $current_page_rules ) ) ) return false;

		// Check each exclusion rule if matches current page.
		// Skip if there is any exclusion rule matched.
		if ( 0 < count( array_intersect( $settings['exclusion_rules'], $current_page_rules ) ) ) return false;

		// Check each user role if matches current user.
		// Skip if there is no user role matched.
		if ( 0 === count( array_intersect( $settings['user_roles'], $current_user_roles ) ) ) return false;

		return true;
	}

	/**
	 * Return all available Blocks.
	 *
	 * @param boolean $flatten
	 * @return array
	 */
	public function get_all_template_hooks( $flatten = false ) {
		$hooks = array(
			esc_attr__( 'Page Canvas', 'eltron-features' ) => array(
				'eltron/frontend/before_canvas' => esc_attr__( 'Before Page Canvas', 'eltron-features' ),
				'eltron/frontend/after_canvas' => esc_attr__( 'After Page Canvas', 'eltron-features' ),
			),
			esc_attr__( 'Header', 'eltron-features' ) => array(
				'eltron/frontend/header' => esc_attr__( 'Replace Header (Desktop & Mobile Header)', 'eltron-features' ),
				'eltron/frontend/before_header' => esc_attr__( 'Before Header', 'eltron-features' ),
				'eltron/frontend/after_header' => esc_attr__( 'After Header', 'eltron-features' ),
			),
			esc_attr__( 'Content & Sidebar', 'eltron-features' ) => array(
				'eltron/frontend/before_primary_and_sidebar' => esc_attr__( 'Before Main Content & Sidebar', 'eltron-features' ),
				'eltron/frontend/before_primary_and_sidebar' => esc_attr__( 'After Main Content & Sidebar', 'eltron-features' ),
				'eltron/frontend/before_main' => esc_attr__( 'Before Main Content', 'eltron-features' ),
				'eltron/frontend/after_main' => esc_attr__( 'After Main Content', 'eltron-features' ),
				'eltron/frontend/before_sidebar' => esc_attr__( 'Before Sidebar', 'eltron-features' ),
				'eltron/frontend/after_sidebar' => esc_attr__( 'After Sidebar', 'eltron-features' ),
			),
			esc_attr__( 'Post / Page Content', 'eltron-features' ) => array(
				'eltron/frontend/entry/before_header' => esc_attr__( 'Before Post Entry Header', 'eltron-features' ),
				'eltron/frontend/entry/header' => esc_attr__( 'Post Entry Header', 'eltron-features' ),
				'eltron/frontend/entry/after_header' => esc_attr__( 'After Post Entry Header', 'eltron-features' ),
				'eltron/frontend/entry/before_content' => esc_attr__( 'Before Post Entry Content', 'eltron-features' ),
				'eltron/frontend/entry/after_content' => esc_attr__( 'After Post Entry Content', 'eltron-features' ),
				'eltron/frontend/entry/before_footer' => esc_attr__( 'Before Post Entry Footer', 'eltron-features' ),
				'eltron/frontend/entry/footer' => esc_attr__( 'Post Entry Footer', 'eltron-features' ),
				'eltron/frontend/entry/after_footer' => esc_attr__( 'After Post Entry Footer', 'eltron-features' ),
			),
			esc_attr__( 'Comments', 'eltron-features' ) => array(
				'eltron/frontend/before_comments' => esc_attr__( 'Before Comments Section', 'eltron-features' ),
				'eltron/frontend/before_comments_list' => esc_attr__( 'Before Comments List', 'eltron-features' ),
				'eltron/frontend/after_comments_list' => esc_attr__( 'After Comments List', 'eltron-features' ),
				'eltron/frontend/after_comments' => esc_attr__( 'After Comments Section', 'eltron-features' ),
			),
			esc_attr__( 'Footer', 'eltron-features' ) => array(
				'eltron/frontend/footer' => esc_attr__( 'Replace Footer (Widgets & Bottom Bar)', 'eltron-features' ),
				'eltron/frontend/before_footer' => esc_attr__( 'Before Footer', 'eltron-features' ),
				'eltron/frontend/after_footer' => esc_attr__( 'After Footer', 'eltron-features' ),
			),
			esc_attr__( 'Others Hooks from Eltron, WordPress core, or Plugins', 'eltron-features' ) => array(
				'custom' => esc_attr__( 'Enter the hook name: ...', 'eltron-features' ),
			),
		);

		$hooks = apply_filters( 'eltron/pro/blocks/hooks', $hooks );

		if ( $flatten ) {
			$hooks = eltron_flatten_array( $hooks );
		}

		return $hooks;
	}

	/**
	 * Return all display rules.
	 *
	 * @param boolean $flatten
	 * @return array
	 */
	public function get_all_display_rules( $flatten = false ) {
		$rules = array();

		$rules[ esc_attr__( 'General', 'eltron-features' ) ] = array(
			'general|global'         => esc_attr__( 'Entire Website', 'eltron-features' ),
			'general|singular'       => esc_attr__( 'All Singulars (any post type)', 'eltron-features' ),
			'general|archive'        => esc_attr__( 'All Archives (any post type)', 'eltron-features' ),
			'general|search'         => esc_attr__( 'Search Results', 'eltron-features' ),
			'general|not_found'      => esc_attr__( 'Not Found (404)', 'eltron-features' ),
			'general|date_archive'   => esc_attr__( 'Date Archive', 'eltron-features' ),
			'general|author_archive' => esc_attr__( 'Author Archive', 'eltron-features' ),
			'general|front_page'     => esc_attr__( 'Front Page', 'eltron-features' ),
		);

		$rules[ esc_attr__( 'Pages', 'eltron-features' ) ][ 'posts|page' ] = esc_html__( 'Static Pages (singular)', 'eltron-features' );

		$post_types = array_merge(
			array( 'post' ),
			get_post_types( array(
				'public'             => true,
				'publicly_queryable' => true,
				'rewrite'            => true,
			), 'names' )
		);
		foreach ( $post_types as $post_type ) {
			$obj = get_post_type_object( $post_type );

			$tax_objects = get_taxonomies( array(
				'object_type' => array( $post_type ),
				'public'      => true,
				'rewrite'     => true,
			), 'objects' );

			// Posts archive.
			/* translators: %s: post type plural label. */
			$rules[ esc_attr( $obj->labels->name ) ][ 'post_type_archive|' . $obj->name ] = sprintf( esc_html__( '%s Archive', 'eltron-features' ), $obj->labels->name );

			foreach ( $tax_objects as $tax_obj ) {
				// Taxonomy archive.
				/* translators: %s: taxonomy singular label. */
				$rules[ esc_attr( $obj->labels->name ) ][ 'tax_archive|' . $tax_obj->name ] = sprintf( esc_html__( '%s Archive', 'eltron-features' ), $tax_obj->labels->singular_name );
			}

			// Posts
			/* translators: %s: post type plural label. */
			$rules[ esc_attr( $obj->labels->name ) ][ 'posts|' . $obj->name ] = sprintf( esc_html__( '%s (singular)', 'eltron-features' ), $obj->labels->name );

			foreach ( $tax_objects as $tax_obj ) {
				// Posts in specific taxonomy
				/* translators: %1$s: post type plural label, %1$s: taxonomy singular label. */
				$rules[ esc_attr( $obj->labels->name ) ][ 'posts_in_tax|' . $tax_obj->name ] = sprintf( esc_html__( '%1$s in %2$s', 'eltron-features' ), $obj->labels->name, $tax_obj->labels->singular_name );
			}
		}

		$rules = apply_filters( 'eltron/pro/blocks/page_rules', $rules );

		if ( $flatten ) {
			$rules = eltron_flatten_array( $rules );
		}

		return $rules;
	}

	/**
	 * Return all user roles.
	 *
	 * @param boolean $flatten
	 * @return array
	 */
	public function get_all_user_roles( $flatten = false ) {
		$specific = array();
		foreach ( get_editable_roles() as $key => $value ) {
			$specific[ $key ] = $value['name'];
		}

		$roles = array(
			esc_attr__( 'Basic', 'eltron-features' ) => array(
				'public'     => esc_attr__( 'All Users', 'eltron-features' ),
				'logged_out' => esc_attr__( 'Logged out Users', 'eltron-features' ),
				'logged_in'  => esc_attr__( 'Logged in Users', 'eltron-features' ),
			),
			esc_attr__( 'Specific Roles', 'eltron-features' ) => $specific,
		);

		$roles = apply_filters( 'eltron/pro/blocks/user_roles', $roles );

		if ( $flatten ) {
			$roles = eltron_flatten_array( $roles );
		}

		return $roles;
	}
}

Eltron_Pro_Module_Custom_Blocks::instance();