<?php
/**
 * Custom Blocks Admin page
 */

// Prevent direct access.
if ( ! defined( 'ABSPATH' ) ) exit;

class Eltron_Pro_Module_Custom_Blocks_Admin {

	/**
	 * Singleton instance
	 *
	 * @var Eltron_Pro_Module_Custom_Blocks_Admin
	 */
	private static $instance;

	/**
	 * ====================================================
	 * Singleton & constructor functions
	 * ====================================================
	 */

	/**
	 * Get singleton instance.
	 *
	 * @return Eltron_Pro_Module_Custom_Blocks_Admin
	 */
	public static function instance() {
		if ( is_null( self::$instance ) ) {
			self::$instance = new self();
		}
		return self::$instance;
	}

	/**
	 * Class constructor
	 */
	protected function __construct() {
		// Admin menu
		add_action( 'eltron/admin/menu', array( $this, 'add_admin_menu' ) );
		
		// List page
		add_filter( 'views_edit-' . Eltron_Pro_Module_Custom_Blocks::POST_TYPE, array( $this, 'render_description' ) );
		add_filter( 'manage_' . Eltron_Pro_Module_Custom_Blocks::POST_TYPE . '_posts_columns', array( $this, 'manage_list_columns' ) );
		add_action( 'manage_' . Eltron_Pro_Module_Custom_Blocks::POST_TYPE . '_posts_custom_column', array( $this, 'manage_list_columns_content' ), 10, 2 );

		// Edit page
		add_action( 'add_meta_boxes', array( $this, 'add_meta_box' ), 10, 2 );
		add_action( 'save_post', array( $this, 'save_meta_box' ) );

		// Page Settings Metabox
		add_action( 'eltron/admin/metabox/page_settings/fields', array( $this, 'add_metabox_page_settings_fields' ), 10, 2 );

		// AJAX callbacks
		add_action( 'wp_ajax_eltron_pro__get_posts_as_select2_choices', array( $this, 'ajax_get_posts_as_select2_choices' ) );
		add_action( 'wp_ajax_eltron_pro__get_terms_as_select2_choices', array( $this, 'ajax_get_terms_as_select2_choices' ) );
	}
	
	/**
	 * ====================================================
	 * Hook functions
	 * ====================================================
	 */

	/**
	 * Add admin submenu page: Appearance > Custom Blocks.
	 */
	public function add_admin_menu() {
		add_theme_page(
			null,
			esc_html__( 'Custom Blocks', 'eltron-features' ),
			'edit_pages',
			'edit.php?post_type=' . Eltron_Pro_Module_Custom_Blocks::POST_TYPE
		);
	}

	/**
	 * Add columns on posts list.
	 *
	 * @param array $columns
	 * @return array
	 */
	public function manage_list_columns( $columns ) {
		$new = array(
			'hook' => esc_html__( 'Attached Hook', 'eltron-features' ),
			'display' => esc_html__( 'Display on', 'eltron-features' ),
			'roles' => esc_html__( 'Visible to', 'eltron-features' ),
			'responsive' => esc_html__( 'Responsive Visibility', 'eltron-features' ),
			'shortcode' => esc_html__( 'Shortcode', 'eltron-features' ),
		);

		$return = array();
		foreach ( $columns as $key => $label ) {
			$return[ $key ] = $label;
			
			if ( 'title' == $key ) {
				$return = array_merge( $return, $new );
			}
		}

		return $return;
	}

	/**
	 * Add columns content on posts list.
	 *
	 * @param array $column_name
	 * @param integer $post_id
	 * @return array
	 */
	public function manage_list_columns_content( $column_name, $post_id ) {
		$settings = Eltron_Pro_Module_Custom_Blocks::instance()->get_block_settings( $post_id );

		switch ( $column_name ) {
			case 'hook':
				$hooks = Eltron_Pro_Module_Custom_Blocks::instance()->get_all_template_hooks( true );

				if ( ! empty( $settings['hook_action'] ) ) {
					if ( 'custom' === $settings['hook_action'] ) {
						printf(
							/* translators: %s: The hook action tag entered by user. */
							esc_html__( 'Custom Hook: %s', 'eltron-features' ),
							$settings['hook_action_custom']
						);
					} else {
						echo esc_html( eltron_array_value( $hooks, $settings['hook_action'] ) );
					}

					echo esc_html( ' (' . $settings['hook_priority'] . ')' );
				}

				break;

			case 'display':
				if ( isset( $settings['display_rules'] ) ) {
					foreach ( $settings['display_rules'] as $rule ) {
						$rule_chunks = explode( ':', $rule );
						echo '<span class="dashicons dashicons-yes"></span>' . $this->_get_display_rule_string( eltron_array_value( $rule_chunks, 0 ), eltron_array_value( $rule_chunks, 1 ) ) . '<br>';
					}

					if ( isset( $settings['exclusion_rules'] ) ) {
						foreach ( $settings['exclusion_rules'] as $rule ) {
							$rule_chunks = explode( ':', $rule );
							echo '<span class="dashicons dashicons-no-alt"></span>' . $this->_get_display_rule_string( eltron_array_value( $rule_chunks, 0 ), eltron_array_value( $rule_chunks, 1 ) ) . '<br>';
						}
					}
				}
				
				break;

			case 'responsive':
				foreach ( array( 'desktop', 'tablet', 'mobile' ) as $device ) {
					$active = in_array( $device, $settings['devices'] );
					echo '<span class="dashicons dashicons-' . ( 'mobile' === $device ? 'smartphone' : $device ) . '" style="opacity:' . ( $active ? '1' : '0.25' ) . ';" title="' . sprintf( $active ? esc_html__( 'Show on %s', 'eltron-features' ) : esc_html__( 'Hide on %s', 'eltron-features' ), $device ) . '"></span>';
				}
				break;

			case 'roles':
				$roles = Eltron_Pro_Module_Custom_Blocks::instance()->get_all_user_roles( true );
				
				if ( isset( $settings['user_roles'] ) ) {
					foreach ( $settings['user_roles'] as $role ) {
						echo $roles[ $role ] . '<br>';
					}
				} else {
					echo '&mdash;';
				}
				break;
			
			case 'shortcode':
				$shortcode = '[' . Eltron_Pro_Module_Custom_Blocks::POST_TYPE . ' id="' . $post_id . '"]';
				echo '<code>' . esc_attr( $shortcode ) . '</code>';
				break;
		}
	}

	/**
	 * Add meta box to Block edit page.
	 *
	 * @param string $post_type
	 * @param WP_Post $post
	 */
	public function add_meta_box( $post_type, $post ) {
		add_meta_box(
			Eltron_Pro_Module_Custom_Blocks::POST_TYPE . '_settings',
			esc_html__( 'Block Settings', 'eltron-features' ),
			array( $this, 'render_meta_box' ),
			Eltron_Pro_Module_Custom_Blocks::POST_TYPE,
			'normal',
			'high'
		);
	}

	/**
	 * Save meta box.
	 *
	 * @param integer $post_id
	 */
	public function save_meta_box( $post_id ) {
		// Check if our nonce is set.
		if ( ! isset( $_POST[ Eltron_Pro_Module_Custom_Blocks::POST_TYPE . '_settings_nonce'] ) ) {
			return;
		}

		// Verify that the nonce is valid.
		if ( ! wp_verify_nonce( $_POST[ Eltron_Pro_Module_Custom_Blocks::POST_TYPE . '_settings_nonce'], Eltron_Pro_Module_Custom_Blocks::POST_TYPE . '_settings' ) ) {
			return;
		}

		// If this is an autosave, our form has not been submitted, so we don't want to do anything.
		if ( defined( 'DOING_AUTOSAVE' ) && DOING_AUTOSAVE ) {
			return;
		}

		// Check the user's permissions.
		if ( ! current_user_can( 'edit_post', $post_id ) ) {
			return;
		}

		$settings = array();

		// Always save display rules, exclusion rules, and user roles even if no value is selected.
		// This is because the jQuery repeater doesn't include empty list as $_POST value.
		$settings = array(
			'display_rules' => array(),
			'exclusion_rules' => array(),
			'user_roles' => array(),
		);

		if ( isset( $_POST[ Eltron_Pro_Module_Custom_Blocks::POST_TYPE . '_settings'] ) ) {
			foreach ( $_POST[ Eltron_Pro_Module_Custom_Blocks::POST_TYPE . '_settings'] as $key => $value ) {
				switch ( $key ) {
					case 'display_rules':
					case 'exclusion_rules':
						$settings[ $key ] = array();

						foreach ( $value as $rule ) {
							if ( '' ===  $rule['type'] ) continue;

							$settings[ $key ][] = $rule['type'] . ( ! empty( $rule['value'] ) ? ':' . $rule['value'] : '' );
						}
						break;

					case 'user_roles':
						$settings[ $key ] = array();

						foreach ( $value as $role ) {
							if ( '' === $role['value'] ) continue;

							$settings[ $key ][] = $role['value'];
						}
						break;

					case 'id':
					case 'class':
						$settings[ $key ] = sanitize_html_class( $value );
						break;

					case 'container_width':
						$settings[ $key ] = absint( $value );
						break;
					
					default:
						if ( '' !== $value ) {
							$settings[ $key ] = $value;
						}
						break;
				}
			}
		}

		// Update the meta field in the database.
		update_post_meta( $post_id, '_' . Eltron_Pro_Module_Custom_Blocks::POST_TYPE . '_settings', $settings );
	}

	/**
	 * Add Page Settings metabox fields.
	 */
	public function add_metabox_page_settings_fields( $obj, $tab ) {
		if ( 'custom-blocks' !== $tab ) {
			return;
		}

		$current_conditions = array();
		$list = array();

		// Check current edited page display conditions.
		if ( is_a( $obj, 'WP_Post' ) ) {
			$current_conditions = Eltron_Pro_Module_Custom_Blocks::instance()->get_current_page_rules__post( $obj );
		}
		elseif ( is_a( $obj, 'WP_Term' ) ) {
			$current_conditions = Eltron_Pro_Module_Custom_Blocks::instance()->get_current_page_rules__term( $obj );
		}

		$hooks = Eltron_Pro_Module_Custom_Blocks::instance()->get_all_template_hooks( true );

		// Iterate through each content block.
		// Check each display rules and attached hook.
		$all_blocks = get_posts( array(
			'post_type' => Eltron_Pro_Module_Custom_Blocks::POST_TYPE,
			'posts_per_page' => -1,
		) );
		foreach ( $all_blocks as $block ) {
			// Get block settings.
			$settings = Eltron_Pro_Module_Custom_Blocks::instance()->get_block_settings( $block );

			// Skip if no action hook found in the block.
			if ( '' === eltron_array_value( $settings, 'hook_action' ) ) {
				continue;
			}

			// Get hook name.
			$hook_name = eltron_array_value( $hooks, eltron_array_value( $settings, 'hook_action' ) );
			if ( 'custom' === $hook_name ) {
				$hook_name .= ': ' . eltron_array_value( $settings, 'hook_action_custom' );
			}

			// Check exclusion rules.
			// Skip if any of current page conditions matches any of the exclusion rules.
			if ( 0 < count( array_intersect( $current_conditions, eltron_array_value( $settings, 'exclusion_rules', array() ) ) ) ) {
				continue;
			}

			// Check display rules.
			// Add to inserted blocks list if any of current page conditions matches any of the display rules.
			if ( 0 < count( array_intersect( $current_conditions, eltron_array_value( $settings, 'display_rules', array() ) ) ) ) {
				$list[] = array(
					'title'         => get_the_title( $block ),
					'hook_name'     => $hook_name,
					'hook_priority' => eltron_array_value( $settings, 'hook_priority', 10 ),
					'edit_link'     => esc_url( add_query_arg( array( 'post' => $block->ID, 'action' => 'edit' ), admin_url( 'post.php' ) ) ),
				);
			}
		}
		?>
		<div class="eltron-admin-form-row eltron-admin-inserted-custom-blocks-list">
			<div class="eltron-admin-form-field">
				<h4><?php esc_html_e( 'List of custom blocks that were inserted to this page via template hooks', 'eltron-features' ); ?></h4>
				<table>
					<thead>
						<tr>
							<th width="70%"><?php esc_html_e( 'Title', 'eltron-features' ); ?></th>
							<th width="30%"><?php esc_html_e( 'Attached Hook', 'eltron-features' ); ?></th>
							<th width="100px"><?php esc_html_e( 'Priority', 'eltron-features' ); ?></th>
						</tr>
					</thead>
					<tbody>
						<?php if ( 1 > count( $list ) ) : ?>
							<tr>
								<td colspan="3"><?php esc_html_e( 'No block inserted to this page yet.', 'eltron-features' ); ?></td>
							</tr>
						<?php else : ?>
							<?php foreach ( $list as $item ) : ?>
								<tr>
									<td>
										<a href="<?php echo esc_url( eltron_array_value( $item, 'edit_link' ) ); ?>"><?php echo esc_html( eltron_array_value( $item, 'title' ) ); ?></a>
									</td>
									<td><?php echo esc_html( eltron_array_value( $item, 'hook_name' ) ); ?></td>
									<td><?php echo esc_html( eltron_array_value( $item, 'hook_priority' ) ); ?></td>
								</tr>
							<?php endforeach; ?>
						<?php endif; ?>
					</tbody>
				</table>
				<p style="margin: 0.75em 0; font-style: italic;"><?php esc_html_e( 'NOTE: Blocks that were added via shortcode are not included in the above list.', 'eltron-features' ); ?></p>
				<p style="margin: 1.25em 0 0;">
					<a href="<?php echo esc_url( add_query_arg( array( 'post_type' => 'eltron_block' ), admin_url( 'edit.php' ) ) ); ?>" class="button button-secondary"><?php esc_html_e( 'Add or Manage Custom Blocks', 'eltron-features' ); ?></a>
				</p>
			</div>
		</div>
		<?php
	}

	/**
	 * ====================================================
	 * Private functions
	 * ====================================================
	 */

	/**
	 * Return a display rule string.
	 *
	 * @param string $type
	 * @param string $value
	 * @return string
	 */
	private function _get_display_rule_string( $type, $value = null ) {
		$rules = Eltron_Pro_Module_Custom_Blocks::instance()->get_all_display_rules( true );

		$type_chunks = explode( '|', $type );

		$label = eltron_array_value( $rules, $type );

		if ( ! empty( $value ) ) {
			switch ( eltron_array_value( $type_chunks, 0 ) ) {
				case 'posts':
					$label .= ': ' . get_the_title( $value );
					break;

				case 'tax_archive':
				case 'posts_in_tax':
					$label .= ': ' . get_term_field( 'name', $value, eltron_array_value( $type_chunks, 1 ) );
					break;
			}
		}

		return $label;
	}

	/**
	 * Return all posts by specified post type as choices.
	 *
	 * @param string $post_type
	 * @return array
	 */
	private function _get_posts_as_choices( $post_type ) {
		$choices = array();

		// Get posts by specified post type.
		$posts = get_posts( array(
			'post_type' => $post_type,
			'posts_per_page' => -1,
			'orderby' => 'title',
			'order' => 'ASC',
		) );

		// Add posts into choices.
		foreach ( $posts as $post ) {
			$choices[ $post->ID ] = get_post_field( 'post_title', $post->ID, 'attribute' );
		}

		return $choices;
	}

	/**
	 * Return all terms by specified taxonomy as choices.
	 *
	 * @param string $taxonomy
	 * @return array
	 */
	private function _get_terms_as_choices( $taxonomy ) {
		$choices = array();

		// Get posts by specified post type.
		$terms = get_terms( array(
			'taxonomy' => $taxonomy,
			'orderby' => 'name',
			'order' => 'ASC',
		) );

		// Add terms into choices.
		foreach ( $terms as $term ) {
			$choices[ $term->term_id ] = get_term_field( 'name', $term->term_id, $taxonomy, 'attribute' );
		}

		return $choices;
	}

	/**
	 * ====================================================
	 * AJAX functions
	 * ====================================================
	 */

	/**
	 * AJAX callback to get all posts by specified post type for select's options on Block Settings meta box.
	 */
	public function ajax_get_posts_as_select2_choices() {
		check_ajax_referer( 'eltron-pro-admin', '_ajax_nonce' );

		if ( ! isset( $_REQUEST['post_type'] ) ) {
			wp_send_json_error( esc_html__( 'No valid post type specified', 'eltron-features' ) );
		}

		// Default choices.
		if ( isset( $_REQUEST['use_all_as_option'] ) && intval( $_REQUEST['use_all_as_option'] ) ) {
			$data = array(
				array(
					'id' => '',
					'text' => esc_attr__( 'All', 'eltron-features' ),
				),
			);
		} else {
			$data = array();
		}

		// Add posts into choices.
		foreach ( $this->_get_posts_as_choices( $_REQUEST['post_type'] ) as $id => $text ) {
			$data[] = array(
				'id'   => $id,
				'text' => $text,
			);
		}

		wp_send_json_success( $data );
	}

	/**
	 * AJAX callback to get all terms by specified taxonomy for select's options on Block Settings meta box.
	 */
	public function ajax_get_terms_as_select2_choices() {
		check_ajax_referer( 'eltron-pro-admin', '_ajax_nonce' );

		if ( ! isset( $_REQUEST['taxonomy'] ) ) {
			wp_send_json_error( esc_html__( 'No valid taxonomy specified', 'eltron-features' ) );
		}

		// Default choices.
		if ( isset( $_REQUEST['use_all_as_option'] ) && intval( $_REQUEST['use_all_as_option'] ) ) {
			$data = array(
				array(
					'id' => '',
					'text' => esc_attr__( 'All', 'eltron-features' ),
				),
			);
		} else {
			$data = array();
		}

		// Add terms into choices.
		foreach ( $this->_get_terms_as_choices( $_REQUEST['taxonomy'] ) as $id => $text ) {
			$data[] = array(
				'id'   => $id,
				'text' => $text,
			);
		}

		wp_send_json_success( $data );
	}

	/**
	 * ====================================================
	 * Render functions
	 * ====================================================
	 */

	/**
	 * Add description on top of the list page.
	 *
	 * @param string $views
	 * @return string
	 */
	public function render_description( $views ) {
		?>
		<div class="notice notice-info">

			<p><?php esc_html_e( 'Custom Block is a reusable and portable content that can be inserted into anywhere on your website via template hooks or shortcode.', 'eltron-features' ); ?> </p>
		</div>
		<?php

		return $views;
	}

	/**
	 * Render settings meta box on Block edit page.
	 *
	 * @param WP_Post $post
	 */
	public function render_meta_box( $post ) {
		$option_key = Eltron_Pro_Module_Custom_Blocks::POST_TYPE . '_settings';

		// Fetch settings values from DB and sanitize them.
		$settings = get_post_meta( $post->ID, '_' . Eltron_Pro_Module_Custom_Blocks::POST_TYPE . '_settings', true );

		$settings = wp_parse_args( $settings, array(
			'hook_action'        => '',
			'hook_action_custom' => '',
			'hook_priority'      => '',

			'display_rules'      => array( 'general|global' ),
			'exclusion_rules'    => array(),
			'user_roles'         => array( 'public' ),
			'devices'            => array( 'desktop', 'tablet', 'mobile' ),

			'tag'                => 'div',
			'id'                 => '',
			'class'              => '',
			'use_container'      => 0,
			'container_width'    => '',
		) );

		// Add a nonce field so we can check for it later.
		wp_nonce_field( Eltron_Pro_Module_Custom_Blocks::POST_TYPE . '_settings', Eltron_Pro_Module_Custom_Blocks::POST_TYPE . '_settings_nonce' );
		?>
		<div id="eltron-metabox-page-settings" class="eltron-admin-metabox-page-settings eltron-admin-metabox eltron-admin-form">
			<ul class="eltron-admin-metabox-nav">
				<li class="eltron-admin-metabox-nav-item active"><a href="#eltron-block-settings--location"><?php esc_html_e( 'Location', 'eltron-features' ); ?></a>
				<li class="eltron-admin-metabox-nav-item"><a href="#eltron-block-settings--rules"><?php esc_html_e( 'Display Rules', 'eltron-features' ); ?></a>
				<li class="eltron-admin-metabox-nav-item"><a href="#eltron-block-settings--markup"><?php esc_html_e( 'Markup & Styles', 'eltron-features' ); ?></a>
			</ul>

			<div class="eltron-admin-metabox-panels">

				<div id="eltron-block-settings--location" class="eltron-admin-metabox-panel active">
					<div class="eltron-admin-form-row">
						<div class="eltron-admin-form-label"><label><?php esc_html_e( 'Attach to hook', 'eltron-features' ); ?></label></div>
						<div class="eltron-admin-form-field">
							<div class="eltron-block-hook-action">
								<span class="eltron-block-hook-action-select">
									<?php
									$key = 'hook_action';
									Eltron_Admin_Fields::render_field( array(
										'name'        => $option_key . '[' . $key . ']',
										'type'        => 'select',
										'value'       => $settings[ $key ],
										'choices'     => array_merge(
											array( '' => esc_html__( '-- None, use via shortocode only --', 'eltron-features' ) ),
											Eltron_Pro_Module_Custom_Blocks::instance()->get_all_template_hooks()
										),
									) );
									?>
								</span>
								<span class="eltron-block-hook-action-custom <?php echo esc_attr( 'custom' === $settings[ $key ] ? 'eltron-show' : 'eltron-hide' ); ?>">
									<?php
									$key = 'hook_action_custom';
									Eltron_Admin_Fields::render_field( array(
										'name'        => $option_key . '[' . $key . ']',
										'type'        => 'text',
										'value'       => $settings[ $key ],
										'placeholder' => esc_attr__( 'e.g. "wp_head" or "woocommerce_before_shop_loop"', 'eltron-features' ),
									) );
									?>
								</span>
								<br><br>
								<span class="eltron-block-hook-priority">
									<strong><?php esc_html_e( 'Priority:', 'eltron-features' ); ?></strong>&nbsp;
									<?php
									$key = 'hook_priority';
									Eltron_Admin_Fields::render_field( array(
										'name'        => $option_key . '[' . $key . ']',
										'type'        => 'number',
										'value'       => $settings[ $key ],
										'min'         => 0,
										'step'        => 1,
										'placeholder' => 10,
										'disabled'    => '' == $settings[ $key ] ? true : false,
									) );
									?>&nbsp;<span class="description"><?php esc_html_e( 'Smaller would be displayed first.', 'eltron-features' ); ?></span>
								</span>
							</div>
						</div>
					</div>

					<hr>

					<div class="eltron-admin-form-row">
						<div class="eltron-admin-form-label"><label><?php esc_html_e( 'Or use via shortcode', 'eltron-features' ); ?></label></div>
						<div class="eltron-admin-form-field">
							<div class="description" style="padding-top: 0.5em;">
								<code id="eltron-block-shortcode"><?php echo esc_attr( $shortcode = '[' . Eltron_Pro_Module_Custom_Blocks::POST_TYPE . ' id="' . $post->ID . '"]' ); ?></code>
							</div>
							<p><?php esc_html_e( 'Copy and paste this shortcode into any content / widget / text editor that supports shortcode.', 'eltron-features' ); ?></p>
						</div>
					</div>
				</div>

				<div id="eltron-block-settings--rules" class="eltron-admin-metabox-panel">
					<?php
					$display_item = function( $rule = '' ) {
						$is_template = false;

						// Check if it's a template
						if ( empty( $rule ) ) {
							$is_template = true;
						}

						// Convert the rule into an array and sanitize it.
						$rule = explode( ':', $rule );

						$is_with_value = is_null( eltron_array_value( $rule, 1 ) ) ? false : true;
						?>
						<li data-repeater-item class="eltron-block-display-rule eltron-admin-repeater-control-item" <?php echo $is_template ? 'style="display: none;"' : ''; ?>>
							<div class="eltron-admin-form-row">
								<div class="eltron-block-display-rule-type eltron-admin-form-field">
									<?php
									Eltron_Admin_Fields::render_field( array(
										'name'        => 'type',
										'type'        => 'select',
										'value'       => eltron_array_value( $rule, 0 ),
										'choices'     => array_merge(
											array( '' => esc_html__( '-- None selected --', 'eltron-features' ) ),
											Eltron_Pro_Module_Custom_Blocks::instance()->get_all_display_rules()
										),
										'class'       => 'widefat',
									) );
									?>
								</div>
								<div class="eltron-block-display-rule-value eltron-admin-form-field <?php echo $is_with_value ? 'eltron-show' : 'eltron-hide'; ?>">
									<?php
									$type = explode( '|', eltron_array_value( $rule, 0 ) );
									$choices = array();

									switch ( eltron_array_value( $type, 0 ) ) {
										case 'posts':
											$choices = array( '' => esc_attr__( 'All', 'eltron-features' ) ) + $this->_get_posts_as_choices( eltron_array_value( $type, 1 ) );
											break;

										case 'tax_archive':
											$choices = array( '' => esc_attr__( 'All', 'eltron-features' ) ) + $this->_get_terms_as_choices( eltron_array_value( $type, 1 ) );
											break;

										case 'posts_in_tax':
											$choices = $this->_get_terms_as_choices( eltron_array_value( $type, 1 ) );
											break;
									}

									Eltron_Admin_Fields::render_field( array(
										'name'        => 'value',
										'type'        => 'select',
										'value'       => eltron_array_value( $rule, 1 ),
										'choices'     => $choices,
										'class'       => 'widefat',
									) );
									?>
								</div>
							</div>

							<a href="javascript:;" data-repeater-delete type="button" class="eltron-admin-repeater-control-delete"><span class="dashicons dashicons-dismiss"></span></a>
						</li>
						<?php
					}
					?>

					<div class="eltron-admin-form-row">
						<div class="eltron-admin-form-label"><label><?php esc_html_e( 'Show on', 'eltron-features' ); ?></label></div>
						<div class="eltron-admin-form-field">
							<?php
							$key = 'display_rules';

							$display_config = array(
								'defaultValues' => array(
									'type'  => '',
									'value' => '',
								),
								'initEmpty' => 1 > count( $settings[ $key ] ) ? true : false,
							);
							?>
							<div class="eltron-block-display-rules eltron-admin-repeater-control eltron-admin-form" data-config="<?php echo esc_attr( json_encode( $display_config ) ); ?>">
								<ul data-repeater-list="<?php echo esc_attr( $option_key . '[' . $key . ']' ); ?>" class="eltron-admin-repeater-control-list">
									<?php if ( 0 < count( $settings[ $key ] ) ) {
										foreach ( $settings[ $key ] as $i => $rule ) {
											$display_item( $rule );
										}
									} else {
										$display_item();
									} ?>
								</ul>
								<input data-repeater-create type="button" class="eltron-admin-repeater-control-add button" value="<?php esc_attr_e( 'Add Rule', 'eltron-features' ); ?>">
							</div>
						</div>
					</div>

					<hr>

					<div class="eltron-admin-form-row">
						<div class="eltron-admin-form-label"><label><?php esc_html_e( 'Hide on', 'eltron-features' ); ?></label></div>
						<div class="eltron-admin-form-field">
							<?php
							$key = 'exclusion_rules';

							$display_config = array(
								'defaultValues' => array(
									'type'  => '',
									'value' => '',
								),
								'initEmpty' => 1 > count( $settings[ $key ] ) ? true : false,
							);
							?>
							<div class="eltron-block-display-rules eltron-admin-repeater-control eltron-admin-form" data-config="<?php echo esc_attr( json_encode( $display_config ) ); ?>">
								<ul data-repeater-list="<?php echo esc_attr( $option_key . '[' . $key . ']' ); ?>" class="eltron-admin-repeater-control-list">
									<?php if ( 0 < count( $settings[ $key ] ) ) {
										foreach ( $settings[ $key ] as $i => $rule ) {
											$display_item( $rule );
										}
									} else {
										$display_item();
									} ?>
								</ul>
								<input data-repeater-create type="button" class="eltron-admin-repeater-control-add button" value="<?php esc_attr_e( 'Add Rule', 'eltron-features' ); ?>">
							</div>
						</div>
					</div>

					<hr>

					<?php
					$roles_item = function( $role = '' ) {
						$is_template = false;

						// Check if it's a template
						if ( empty( $role ) ) {
							$is_template = true;
						}
						?>
						<li data-repeater-item class="eltron-block-user-role eltron-admin-repeater-control-item" <?php echo $is_template ? 'style="display: none;"' : ''; ?>>
							<div class="eltron-admin-form-row">
								<div class="eltron-admin-form-field">
									<?php Eltron_Admin_Fields::render_field( array(
										'name'        => 'value',
										'type'        => 'select',
										'value'       => $role,
										'choices'     => array_merge(
											array( '' => esc_html__( '-- None selected --', 'eltron-features' ) ),
											Eltron_Pro_Module_Custom_Blocks::instance()->get_all_user_roles()
										),
										'class'       => 'widefat',
									) ); ?>
								</div>
							</div>

							<a href="javascript:;" data-repeater-delete type="button" class="eltron-admin-repeater-control-delete"><span class="dashicons dashicons-dismiss"></span></a>
						</li>
						<?php
					}
					?>

					<div class="eltron-admin-form-row">
						<div class="eltron-admin-form-label"><label><?php esc_html_e( 'Visible to', 'eltron-features' ); ?></label></div>
						<div class="eltron-admin-form-field">
							<?php
							$key = 'user_roles';
							
							$roles_config = array(
								'defaultValues' => array(),
								'initEmpty' => 1 > count( $settings[ $key ] ) ? true : false,
							);
							?>
							<div class="eltron-block-user-roles eltron-admin-repeater-control eltron-admin-form" data-config="<?php echo esc_attr( json_encode( $roles_config ) ); ?>">
								<ul data-repeater-list="<?php echo esc_attr( $option_key . '[' . $key . ']' ); ?>" class="eltron-admin-repeater-control-list">
									<?php if ( 0 < count( $settings[ $key ] ) ) {
										foreach ( $settings[ $key ] as $i => $role ) {
											$roles_item( $role );
										}
									} else {
										$roles_item();
									} ?>
								</ul>
								<input data-repeater-create type="button" class="eltron-admin-repeater-control-add button" value="<?php esc_attr_e( 'Add Role', 'eltron-features' ); ?>">
							</div>
						</div>
					</div>

					<hr>

					<div class="eltron-admin-form-row">
						<div class="eltron-admin-form-label"><label><?php esc_html_e( 'Responsive Display', 'eltron-features' ); ?></label></div>
						<div class="eltron-admin-form-field">
							<fieldset>
								<?php
								$key = 'devices';

								$devices = array(
									'desktop' => esc_html__( 'Desktop', 'eltron-features' ),
									'tablet' => esc_html__( 'Tablet', 'eltron-features' ),
									'mobile' => esc_html__( 'Mobile', 'eltron-features' ),
								);

								foreach( $devices as $device => $label ) :
									Eltron_Admin_Fields::render_field( array(
										'type'         => 'checkbox',
										'name'         => $option_key . '[' . $key . '][]',
										'value'        => in_array( $device, $settings[ $key ] ) ? $device : '',
										'return_value' => $device,
										'label'        => $label,
										'wrapper'      => false,
									) );
									echo '<br>';
								endforeach;
								?>
							</fieldset>
						</div>
					</div>

					<script type="text/javascript">
						(function($) {
							$( '#eltron-metabox-page-settings' ).on( 'eltron-admin-metabox.ready', function() {
								var $this = $( this ),
								    displayRuleValueOptionsCache = {};

								$this.find( '.eltron-block-display-rules' ).each(function() {
									var $repeater = $( this );

									var initSelect2 = function() {
										$repeater.find( '.eltron-block-display-rule select' ).select2( { width: '100%' } );
									};
									$repeater.on( 'eltron-admin-repeater.ready', initSelect2 );
									$repeater.on( 'eltron-admin-repeater.itemAdded', initSelect2 );

									// Display rules specific value
									var changeDisplayRuleType = function() {
										var $typeSelect = $( this ),
										    $rule = $typeSelect.closest( '.eltron-block-display-rule' ),
										    $valueCol = $rule.find( '.eltron-block-display-rule-value' ),
										    $valueSelect = $valueCol.find( 'select' ),
										    type = this.value,
										    typeChunks = type.split( '|' );

										// Hide value field on general and post type archive types.
										if ( 'general' === typeChunks[0] || 'post_type_archive' === typeChunks[0] || '' === typeChunks[0] ) {
											$valueCol.removeClass( 'eltron-show' ).addClass( 'eltron-hide' );
											$valueSelect.val( '' ).prop( 'disabled', true );

											// Reset options of value field.
											$valueSelect.empty();
										}
										// Show value field on other types.
										else {
											$valueCol.removeClass( 'eltron-hide' ).addClass( 'eltron-show' );
											$valueSelect.val( '' ).prop( 'disabled', false );
											
											// Check if options for this type is already cached before.
											if ( displayRuleValueOptionsCache[ type ] ) {
												// Use the cache.
												$valueSelect.empty().select2({
													data: displayRuleValueOptionsCache[ type ],
												});
											}
											// Not yet cached before.
											else {
												var data = {
													_ajax_nonce: eltronProAdminData.ajaxNonce,
												};

												// Fetch the options using AJAX.
												switch ( typeChunks[0] ) {
													case 'posts':
														data.action = 'eltron_pro__get_posts_as_select2_choices';
														data.post_type = typeChunks[1];
														data.use_all_as_option = 1;
														break;

													case 'tax_archive':
														data.action = 'eltron_pro__get_terms_as_select2_choices';
														data.taxonomy = typeChunks[1];
														data.use_all_as_option = 1;
														break;

													case 'posts_in_tax':
														data.action = 'eltron_pro__get_terms_as_select2_choices';
														data.taxonomy = typeChunks[1];
														data.use_all_as_option = 0;
														break;
												}

												$valueSelect.prop( 'disabled', true );

												$.ajax({
													method: 'POST',
													url: ajaxurl,
													cache: false,
													data: data,
												})
												.done(function( response, status, XHR ) {
													if ( response.success ) {
														// Set returned data as select's options.
														$valueSelect.empty().select2({
															data: response.data,
														});

														// Cache options.
														displayRuleValueOptionsCache[ type ] = response.data;
													} else {
														alert( response.data );
													}

													$valueSelect.prop( 'disabled', false );
												});
											}
										}
									};
									$repeater.on( 'change', '.eltron-block-display-rule-type select', changeDisplayRuleType );
								});

								$this.find( '.eltron-block-hook-action' ).each( function() {
									var $row = $( this ),
									    $hookSelect = $row.find( '.eltron-block-hook-action-select select' ),
									    $customCol = $row.find( '.eltron-block-hook-action-custom' );

									$hookSelect.on( 'change', function() {
										if ( 'custom' === this.value ) {
											$customCol.removeClass( 'eltron-hide' ).addClass( 'eltron-show' );
										} else {
											$customCol.removeClass( 'eltron-show' ).addClass( 'eltron-hide' );
										}
									});
								});
								
								$this.find( '.eltron-block-user-roles' ).each(function() {
									var $repeater = $( this );

									var initSelect2 = function() {
										$repeater.find( '.eltron-block-user-role select' ).select2( { width: '100%' } );
									};
									$repeater.on( 'eltron-admin-repeater.ready', initSelect2 );
									$repeater.on( 'eltron-admin-repeater.itemAdded', initSelect2 );
								});
							});

						})(jQuery);
					</script>
				</div>

				<div id="eltron-block-settings--markup" class="eltron-admin-metabox-panel">
					<div class="eltron-admin-form-row">
						<div class="eltron-admin-form-label"><label><?php esc_html_e( 'Tag', 'eltron-features' ); ?></label></div>
						<div class="eltron-admin-form-field">
							<?php
							$key = 'tag';
							Eltron_Admin_Fields::render_field( array(
								'name'        => $option_key . '[' . $key . ']',
								'type'        => 'select',
								'value'       => $settings[ $key ],
								'choices'     => array(
									'div',
									'section',
									'header',
									'footer',
									'aside',
									'nav',
								),
							) );
							?>
						</div>
					</div>

					<div class="eltron-admin-form-row">
						<div class="eltron-admin-form-label"><label><?php esc_html_e( 'Tag ID', 'eltron-features' ); ?></label></div>
						<div class="eltron-admin-form-field">
							<?php
							$key = 'id';
							Eltron_Admin_Fields::render_field( array(
								'name'        => $option_key . '[' . $key . ']',
								'type'        => 'text',
								'value'       => $settings[ $key ],
							) );
							?>
						</div>
					</div>

					<div class="eltron-admin-form-row">
						<div class="eltron-admin-form-label"><label><?php esc_html_e( 'Tag class', 'eltron-features' ); ?></label></div>
						<div class="eltron-admin-form-field">
							<?php
							$key = 'class';
							Eltron_Admin_Fields::render_field( array(
								'name'        => $option_key . '[' . $key . ']',
								'type'        => 'text',
								'value'       => $settings[ $key ],
							) );
							?>
						</div>
					</div>

					<hr>

					<div class="eltron-admin-form-row">
						<div class="eltron-admin-form-label"><label><?php esc_html_e( 'Center content container', 'eltron-features' ); ?></label></div>
						<div class="eltron-admin-form-field">
							<?php
							$key = 'use_container';
							Eltron_Admin_Fields::render_field( array(
								'type'         => 'checkbox',
								'name'         => $option_key . '[' . $key . ']',
								'value'        => $settings[ $key ],
								'return_value' => 1,
								'label'        => esc_html__( 'Enabled', 'eltron-features' ),
							) );
							?>
							<p class="description"><?php esc_html_e( 'If you are using Page Builder to build the content, this option is most likely not needed.', 'eltron-features' ); ?></p>
						</div>
					</div>

					<div class="eltron-admin-form-row">
						<div class="eltron-admin-form-label"><label><?php esc_html_e( 'Container width', 'eltron-features' ); ?></label></div>
						<div class="eltron-admin-form-field">
							<?php
							$key = 'container_width';
							Eltron_Admin_Fields::render_field( array(
								'type'         => 'number',
								'name'         => $option_key . '[' . $key . ']',
								'placeholder'  => intval( eltron_get_theme_mod( 'container_width' ) ),
								'min'          => 600,
								'max'          => 1600,
								'step'         => 1,
							) );
							esc_html_e( 'px', 'eltron-features' );
							?>
						</div>
					</div>
				</div>
			</div>
		</div>
		<?php
	}
}

Eltron_Pro_Module_Custom_Blocks_Admin::instance();